const { db } = require("../db/index");
const { notifications } = require("../db/schemas/notificationsSchema");

const ALLOWED_STATUSES = new Set(["pending", "sent", "failed"]);

const isBlank = (v) => v === null || v === undefined || String(v).trim() === "";

const toRequiredString = (v) => {
  if (isBlank(v)) return null;
  return String(v).trim();
};

const toNullableString = (v) => {
  if (v === null || v === undefined) return null;
  const s = String(v).trim();
  return s === "" ? null : s;
};

const toNullableInt = (v) => {
  if (v === null || v === undefined || v === "") return null;
  const n = typeof v === "number" ? v : parseInt(String(v), 10);
  return Number.isInteger(n) ? n : null;
};

const normalizeNotificationInsert = (input, { defaultSenderId } = {}) => {
  const entityType = toRequiredString(input?.entityType);
  const entityId = toRequiredString(input?.entityId);
  const type = toRequiredString(input?.type);
  const recipientId = toNullableInt(input?.recipientId);

  if (!entityType || !entityId || !type || !recipientId) {
    return { ok: false, value: null };
  }

  const senderIdRaw = toNullableInt(input?.senderId);
  const senderId = senderIdRaw ?? toNullableInt(defaultSenderId);

  const rawStatus = toNullableString(input?.status) || "pending";
  const status = ALLOWED_STATUSES.has(rawStatus) ? rawStatus : "pending";

  return {
    ok: true,
    value: {
      entityType,
      entityId,
      type,
      recipientId,
      senderId,
      title: toNullableString(input?.title),
      message: toNullableString(input?.message),
      payload: input?.payload === undefined ? null : input?.payload,
      status,
      readAt: input?.readAt ?? null,
    },
  };
};

const createNotificationService = async (notificationInput, options = {}) => {
  try {
    const normalized = normalizeNotificationInsert(notificationInput, {
      defaultSenderId: options?.senderId,
    });

    if (!normalized.ok) {
      return {
        success: false,
        message: "Missing required fields",
        data: null,
      };
    }

    const result = await db.insert(notifications).values(normalized.value).$returningId();

    if (!Array.isArray(result) || result.length === 0 || !result[0]?.id) {
      return { success: false, message: "Failed to create notification", data: null };
    }

    return {
      success: true,
      message: "Notification created successfully",
      data: result[0],
    };
  } catch (e) {
    return {
      success: false,
      message: e.cause?.sqlMessage || e.message || "Failed to create notification",
      data: null,
    };
  }
};

const createNotificationsForRecipientsService = async (
  notificationInput,
  recipientIds,
  options = {},
) => {
  try {
    const list = Array.isArray(recipientIds) ? recipientIds : [];
    const cleanRecipientIds = list
      .map((id) => toNullableInt(id))
      .filter((id) => Number.isInteger(id));

    if (cleanRecipientIds.length === 0) {
      return { success: false, message: "No recipient IDs provided", data: null };
    }

    const normalized = normalizeNotificationInsert(
      { ...notificationInput, recipientId: cleanRecipientIds[0] },
      { defaultSenderId: options?.senderId },
    );

    if (!normalized.ok) {
      return {
        success: false,
        message: "Missing required fields",
        data: null,
      };
    }

    const rowsToInsert = cleanRecipientIds.map((rid) => ({
      ...normalized.value,
      recipientId: rid,
    }));

    const result = await db.insert(notifications).values(rowsToInsert).$returningId();

    return {
      success: true,
      message: "Notifications created successfully",
      data: result,
    };
  } catch (e) {
    return {
      success: false,
      message: e.cause?.sqlMessage || e.message || "Failed to create notifications",
      data: null,
    };
  }
};

module.exports = {
  createNotificationService,
  createNotificationsForRecipientsService,
};
