const { eq } = require("drizzle-orm");
const { db } = require("../db/index");
const { users } = require("../db/schemas/userSchema");
const { hashPassword } = require("../utils/passwordUtils");

// just get/fetch all the users with out any filters
const getAllUsersService = async () => {
  try {
    // fetch all the users from the database
    const allUsers = await db.select().from(users);
    // console.log("All Users: ", allUsers);

    // error case: no users found
    if (allUsers.length === 0) {
      return {
        success: false,
        message: "No users found",
        data: null,
      };
    }

    return {
      success: true,
      message: "Users fetched successfully",
      data: allUsers,
    };

  } catch (e) {
    return {
      success: false,
      message: "Failed to fetch users",
      data: null,
    };
  }
};

// save single user details to database

const saveUserService = async (userData) => {
  try {
    // extract password so that it can be hashed before saving

    const { password,...restFields } = userData;

    const hashedPassword = await hashPassword(password);

    const newUserData = {
      ...restFields,
      password: hashedPassword,
    };

    const result = await db.insert(users).values(newUserData).$returningId();

    if (result.length === 0 && !result[0].id) {
      console.log("Failed to insert user");
      return {
        success: false,
        message: "Failed to save user",
        data: null,
      };
    }

    return {
      success: true,
      message: "User saved successfully",
      data: result,
    };
  } catch (e) {
    return {
      success: false,
      message: e.cause.sqlMessage || e.message || "Failed to save user",
      data: null,
    };
  }
};

// update User

const updateUserService = async (userData) => {
  try {
    const { id, ...restFields } = userData;

    // first extract the password if updated then hash it
    const { password } = restFields;

    if (password) {
      const hashedPassword = await hashPassword(password);
      restFields.password = hashedPassword;
    }

    const result = await db
      .update(users)
      .set(restFields)
      .where(eq(users.id, id));

    if (result.affectedRows === 0) {
      return {
        success: false,
        message: "No user found with the given ID",
        data: null,
      };
    } else {
      return {
        success: true,
        message: "User updated successfully",
        data: null,
      };
    }
  } catch (e) {
    return {
      success: false,
      message: e.cause.sqlMessage || e.message || "Failed to update user",
      data: null,
    };
  }
};

const deleteUserService = async (userId) => {
  if (!userId || !Number.isInteger(userId)) {
    return {
      success: false,
      message: "Invalid or missing user ID",
      data: null,
    };
  }

  try {
    const result = await db.delete(users).where(eq(users.id, userId));
    console.log("🚀 ~ deleteUserService ~ result:", result);

    if (result.affectedRows === 0) {
      return {
        success: false,
        message: "No user found with the given ID",
        data: null,
      };
    } else {
      return {
        success: true,
        message: "User deleted successfully",
        data: null,
      };
    }
  } catch (e) {
    return {
      success: false,
      message: e.cause.sqlMessage || e.message || "Failed to delete user",
      data: null,
    };
  }
};


const toggleIsActiveUserService = async (userId, status) =>{
  if (!userId || !Number.isInteger(userId)) {
    return {
      success: false,
      message: "Invalid or missing user ID",
      data: null,
    };
  }
  try{

    const result = await db.update(users).set(
      {
        isActive : status
      }
    ).where(eq(users.id , userId));


    if (result.affectedRows === 0) {
      return {
        success: false,
        message: "No user found with the given ID",
        data: null,
      };
    } else {
      return {
        success: true,
        message: "User Status updated successfully",
        data: null,
      };
    }


  }
  catch(e)
  {
    return {
      success: false,
      message: e.cause.sqlMessage || e.message || "Failed to update status",
      data: null,
    };
  }

}

module.exports = {
  getAllUsersService,
  saveUserService,
  updateUserService,
  deleteUserService,
  toggleIsActiveUserService
};
