const { eq } = require("drizzle-orm");
const { db } = require("../db/index");
const { roles } = require("../db/schemas/roleSchema");

// just get/fetch all the roles without any filters
const getAllRolesService = async () => {
  try {
    // fetch all the roles from the database
    const allRoles = await db.select().from(roles);
    // console.log("All Roles: ", allRoles);

    // error case: no roles found
    if (!allRoles) {
      return {
        success: false,
        message: "No roles found",
        data: null,
      };
    }

    return {
      success: true,
      message: "Roles fetched successfully",
      data: allRoles,
    };
  } catch (e) {
    return {
      success: false,
      message: "Failed to fetch roles",
      data: null,
    };
  }
};

// save single role details to database
const saveRoleService = async (roleData) => {
  try {
    // Check if trying to create a superadmin role when one already exists
    const roleName = String(roleData?.name || "").toLowerCase();
    if (roleName === "superadmin") {
      const existingSuperAdmin = await db.select().from(roles).where(eq(roles.name, "superadmin"));
      if (existingSuperAdmin.length > 0) {
        return {
          success: false,
          message: "A superadmin role already exists. Only one superadmin role is allowed.",
          data: null,
        };
      }
    }

    const result = await db.insert(roles).values(roleData).$returningId();

    if (result.length === 0 && !result[0].id) {
      console.log("Failed to insert role");
      return {
        success: false,
        message: "Failed to save role",
        data: null,
      };
    }

    return {
      success: true,
      message: "Role saved successfully",
      data: result,
    };
  } catch (e) {
    return {
      success: false,
      message: e.cause.sqlMessage || e.message || "Failed to save role",
      data: null,
    };
  }
};

// update Role
const updateRoleService = async (roleData) => {
  try {
    const { id, ...restFields } = roleData;

    // Check if trying to change role name to superadmin when one already exists
    const newRoleName = String(restFields?.name || "").toLowerCase();
    if (newRoleName === "superadmin") {
      const existingSuperAdmin = await db.select().from(roles).where(eq(roles.name, "superadmin"));
      // Check if the existing superadmin is a different role (not the one being updated)
      if (existingSuperAdmin.length > 0 && existingSuperAdmin[0].id !== id) {
        return {
          success: false,
          message: "A superadmin role already exists. Only one superadmin role is allowed.",
          data: null,
        };
      }
    }

    const result = await db
      .update(roles)
      .set(restFields)
      .where(eq(roles.id, id));

    if (result.affectedRows === 0) {
      return {
        success: false,
        message: "No role found with the given ID",
        data: null,
      };
    } else {
      return {
        success: true,
        message: "Role updated successfully",
        data: null,
      };
    }
  } catch (e) {
    return {
      success: false,
      message: e.cause.sqlMessage || e.message || "Failed to update role",
      data: null,
    };
  }
};

const deleteRoleService = async (roleId) => {
  if (!roleId || !Number.isInteger(roleId)) {
    return {
      success: false,
      message: "Invalid or missing role ID",
      data: null,
    };
  }

  try {
    const result = await db.delete(roles).where(eq(roles.id, roleId));
    console.log("🚀 ~ deleteRoleService ~ result:", result);

    if (result.affectedRows === 0) {
      return {
        success: false,
        message: "No role found with the given ID",
        data: null,
      };
    } else {
      return {
        success: true,
        message: "Role deleted successfully",
        data: null,
      };
    }
  } catch (e) {
    return {
      success: false,
      message: e.cause.sqlMessage || e.message || "Failed to delete role",
      data: null,
    };
  }
};

module.exports = {
  getAllRolesService,
  saveRoleService,
  updateRoleService,
  deleteRoleService,
};