const { eq, desc, and, gte, lte } = require("drizzle-orm");


const dayjs = require('dayjs');
const utc = require('dayjs/plugin/utc');

const { db } = require("../db/index");
const {
  form297Tickets,
  form297EquipmentEntries,
  form297PersonnelEntries,
} = require("../db/schemas/form297Schema");
const { allForms } = require("../db/schemas/allForms");

const isBlank = (v) => v === null || v === undefined || String(v).trim() === "";

const toRequiredString = (v) => {
  if (isBlank(v)) return null;
  return String(v).trim();
};

const normalizeUnitType = (v) =>
  String(v).toLowerCase() === "miles" ? "miles" : "hours";

const normalizeTransportRetained = (v) => {
  const s = String(v || "").toLowerCase();
  if (s === "yes") return "yes";
  if (s === "no") return "no";
  return "na";
};

const calcHoursFromTimes = (start, stop) => {
  if (!start || !stop) return null;
  const sm = /^\s*(\d{1,2}):(\d{2})(?::(\d{2}))?\s*$/.exec(String(start));
  const em = /^\s*(\d{1,2}):(\d{2})(?::(\d{2}))?\s*$/.exec(String(stop));
  if (!sm || !em) return null;
  const sh = Number(sm[1]);
  const smin = Number(sm[2]);
  const eh = Number(em[1]);
  const emin = Number(em[2]);
  if ([sh, smin, eh, emin].some((n) => Number.isNaN(n))) return null;
  let startMinutes = sh * 60 + smin;
  let stopMinutes = eh * 60 + emin;
  if (stopMinutes < startMinutes) stopMinutes += 24 * 60;
  const diff = stopMinutes - startMinutes;
  return Math.round((diff / 60) * 100) / 100;
};

const toNullableDecimal = (v) => {
  if (v === null || v === undefined || v === "") return null;
  const n = Number(v);
  if (Number.isNaN(n)) return null;
  return n;
};

const normalizeEquipmentEntries = (entries, unitType) => {
  const list = Array.isArray(entries) ? entries : [];
  return list
    .filter((e) => e && e.workDate)
    .map((e) => {
      const startTime = e.startTime || null;
      const stopTime = e.stopTime || null;
      const startOdometer = toNullableDecimal(e.startOdometer);
      const stopOdometer = toNullableDecimal(e.stopOdometer);

      let total = toNullableDecimal(e.total);
      if (total === null) {
        if (
          unitType === "miles" &&
          startOdometer !== null &&
          stopOdometer !== null
        ) {
          total = Math.round((stopOdometer - startOdometer) * 100) / 100;
        } else {
          const h = calcHoursFromTimes(startTime, stopTime);
          if (h !== null) total = h;
        }
      }

      return {
        workDate: e.workDate,
        startTime,
        stopTime,
        startOdometer,
        stopOdometer,
        total,
        quantity: toNullableDecimal(e.quantity),
        type: e.type || null,
        note: e.note || null,
      };
    });
};

const normalizePersonnelEntries = (entries) => {
  const list = Array.isArray(entries) ? entries : [];
  return list
    .filter((e) => e && e.workDate)
    .map((e) => {
      const startTime1 = e.startTime1 || null;
      const stopTime1 = e.stopTime1 || null;
      const startTime2 = e.startTime2 || null;
      const stopTime2 = e.stopTime2 || null;

      let total = toNullableDecimal(e.total);
      if (total === null) {
        const h1 = calcHoursFromTimes(startTime1, stopTime1) || 0;
        const h2 = calcHoursFromTimes(startTime2, stopTime2) || 0;
        const sum = h1 + h2;
        total = sum ? Math.round(sum * 100) / 100 : null;
      }

      return {
        workDate: e.workDate,
        operatorName: e.operatorName || null,
        operatorUserId: e.operatorUserId ? Number(e.operatorUserId) : null,
        startTime1,
        stopTime1,
        startTime2,
        stopTime2,
        total,
        note: e.note || null,
      };
    });
};

const getAllForm297TicketsService = async (user) => {
  // console.log("🚀 ~ getAllForm297TicketsService ~ user:", user)

  try {
    const userId = user?.id;
    const isAdmin = user?.isAdmin;

    if (!isAdmin && !userId) {
      return {
        success: false,
        message: "Unauthorized",
        data: null,
      };
    }

    let query = db.select().from(form297Tickets);
    if (!isAdmin) {
      query = query.where(eq(form297Tickets.createdBy, userId));
    }
    const rows = await query.orderBy(form297Tickets.id);

    return {
      success: true,
      message: "Shift tickets fetched successfully",
      data: rows,
    };
  } catch (e) {
    return {
      success: false,
      message: "Failed to fetch shift tickets",
      data: null,
    };
  }
};

const getForm297TicketByIdService = async (id, user) => {
  if (!id || !Number.isInteger(id)) {
    return {
      success: false,
      message: "Invalid or missing ticket ID",
      data: null,
    };
  }

  try {
    const userId = user?.id;
    const isAdmin = user?.isAdmin;

    if (!isAdmin && !userId) {
      return {
        success: false,
        message: "Unauthorized",
        data: null,
      };
    }

    const ticket = await db
      .select()
      .from(form297Tickets)
      .where(eq(form297Tickets.id, id))
      .limit(1);
    if (!ticket || ticket.length === 0) {
      return { success: false, message: "Ticket not found", data: null };
    }

    const row = ticket[0];

    if (!isAdmin && row?.createdBy !== userId) {
      return { success: false, message: "Forbidden", data: null };
    }

    const equipment = await db
      .select()
      .from(form297EquipmentEntries)
      .where(eq(form297EquipmentEntries.ticketId, id));

    const personnel = await db
      .select()
      .from(form297PersonnelEntries)
      .where(eq(form297PersonnelEntries.ticketId, id));

    return {
      success: true,
      message: "Ticket fetched successfully",
      data: {
        ticket: row,
        equipmentEntries: equipment,
        personnelEntries: personnel,
      },
    };
  } catch (e) {
    return { success: false, message: "Failed to fetch ticket", data: null };
  }
};

dayjs.extend(utc);
const createForm297TicketService = async (payload, userId) => {
  console.log("createForm297TicketService payload:", payload);
  const alreadyExists = await db
    .select()
    .from(form297Tickets)
    .where(eq(form297Tickets.createdBy, userId))
    .orderBy(desc(form297Tickets.createdAt))
    .limit(1);

  const existing = alreadyExists[0];

  if (existing) {
    // console.log("Found existing: ", alreadyExists);
    // const created = new Date(existing.createdAt);
    // console.log("Creatd:  ", created);
    // const today = new Date();

    // const isToday =
    //   created.getUTCFullYear() === today.getUTCFullYear() &&
    //   created.getUTCMonth() === today.getUTCMonth() &&
    //   created.getUTCDate() === today.getUTCDate();

    const created = dayjs(existing.createdAt).utc();
    const today = dayjs().utc();
    const isToday = created.isSame(today, 'day');


    console.log("created", created, today, isToday)

    if (isToday) {

      return {
        success: false,
        message: `You already created a new ticket today. You can create a new ticket tomorrow.`,
        existingId: existing.id,
      };
    }

    // ELSE → allow creation
  }


  try {
    const agreementNumber = toRequiredString(payload?.agreementNumber);
    const contractorAgencyName = toRequiredString(
      payload?.contractorAgencyName,
    );
    const resourceOrderNumber = toRequiredString(payload?.resourceOrderNumber);
    const incidentName = toRequiredString(payload?.incidentName);
    const incidentNumber = toRequiredString(payload?.incidentNumber);
    const financialCode = toRequiredString(payload?.financialCode);

    if (
      !agreementNumber ||
      !contractorAgencyName ||
      !resourceOrderNumber ||
      !incidentName ||
      !incidentNumber ||
      !financialCode
    ) {
      return { success: false, message: "Missing required fields", data: null };
    }

    const unitType = normalizeUnitType(payload?.unitType);
    const equipmentEntries = normalizeEquipmentEntries(
      payload?.equipmentEntries,
      unitType,
    );
    const personnelEntries = normalizePersonnelEntries(
      payload?.personnelEntries,
    );

    const contractorRepPrintedName = isBlank(payload?.contractorRepPrintedName)
      ? null
      : String(payload.contractorRepPrintedName).trim();
    const contractorRepSignature = isBlank(payload?.contractorRepSignature)
      ? null
      : String(payload.contractorRepSignature).trim();
    const incidentSupervisorPrintedName = isBlank(
      payload?.incidentSupervisorPrintedName,
    )
      ? null
      : String(payload.incidentSupervisorPrintedName).trim();

      const incidentSupervisorPosition = isBlank(payload?.incidentSupervisorPosition,) ? null : String(payload.incidentSupervisorPosition);

    const incidentSupervisorResourceOrderNumber = isBlank(
      payload?.incidentSupervisorResourceOrderNumber,
    )
      ? null
      : String(payload.incidentSupervisorResourceOrderNumber).trim();
    const incidentSupervisorSignature = isBlank(
      payload?.incidentSupervisorSignature,
    )
      ? null
      : String(payload.incidentSupervisorSignature).trim();
      

    const ticketInsert = {
      agreementNumber,
      contractorAgencyName,
      resourceOrderNumber,
      incidentName,
      incidentNumber,
      financialCode,
      equipmentMakeModel: payload.equipmentMakeModel || null,
      equipmentType: payload.equipmentType || null,
      serialVinNumber: payload.serialVinNumber || null,
      licenseIdNumber: payload.licenseIdNumber || null,
      transportRetained: normalizeTransportRetained(payload.transportRetained),
      isFirstLastTicket: !!payload.isFirstLastTicket,
      mobilization: !!payload.mobilization,
      demobilization: !!payload.demobilization,
      unitType,
      remarks: payload.remarks || null,

      contractorRepPrintedName,
      contractorRepSignature,
      incidentSupervisorPrintedName,
      incidentSupervisorPosition, //added new field
      incidentSupervisorResourceOrderNumber,
      incidentSupervisorSignature,

      createdBy: userId || null,
      updatedBy: userId || null,
    };

    const result = await db.transaction(async (tx) => {
      // first I want to create the form if this is new form then create an entry in the allForms table

      let [form] = await tx
        .select()
        .from(allForms)
        .where(eq(allForms.formName, "form297"))
        .limit(1);

      if (!form) {
        const result = await tx.insert(allForms).values({
          formName: "form297",
          formDescription: "Form 297",
        });

        const insertId = result[0].insertId;

        form = { id: insertId };
      }

      const ids = await tx
        .insert(form297Tickets)
        .values({ ...ticketInsert, formId: form.id })
        .$returningId();
      const ticketId = ids?.[0]?.id;

      if (!ticketId) {
        return { ticketId: null };
      }

      if (equipmentEntries.length) {
        await tx
          .insert(form297EquipmentEntries)
          .values(equipmentEntries.map((e) => ({ ...e, ticketId })));
      }

      if (personnelEntries.length) {
        await tx
          .insert(form297PersonnelEntries)
          .values(personnelEntries.map((p) => ({ ...p, ticketId })));
      }

      return { ticketId };
    });

    if (!result.ticketId) {
      return { success: false, message: "Failed to create ticket", data: null };
    }

    return {
      success: true,
      message: "Ticket created successfully",
      data: { id: result.ticketId },
    };
  } catch (e) {
    return {
      success: false,
      message: e?.message || "Failed to create ticket",
      data: null,
    };
  }
};

const updateForm297TicketService = async (payload, user) => {

  // console.log("Hit" , user)
  const id = payload?.id;
  if (!id || !Number.isInteger(id)) {
    return {
      success: false,
      message: "Invalid or missing ticket ID",
      data: null,
    };
  }

  try {
    const existing = await db
      .select()
      .from(form297Tickets)
      .where(eq(form297Tickets.id, id))
      .limit(1);
    if (!existing || existing.length === 0) {
      return { success: false, message: "Ticket not found", data: null };
    }

    if (existing[0].status !== "draft") {
      return {
        success: false,
        message: "Only draft tickets can be updated",
        data: null,
      };
    }

    const isOwner = existing?.[0]?.createdBy === user?.userId;
    const isAdmin = user?.isAdmin;

    if (!isOwner && !isAdmin) {
      console.log("isAdmin", isAdmin)
      return { success: false, message: "Forbidden", data: null };
    }

    const agreementNumber = toRequiredString(payload?.agreementNumber);
    const contractorAgencyName = toRequiredString(
      payload?.contractorAgencyName,
    );
    const resourceOrderNumber = toRequiredString(payload?.resourceOrderNumber);
    const incidentName = toRequiredString(payload?.incidentName);
    const incidentNumber = toRequiredString(payload?.incidentNumber);
    const financialCode = toRequiredString(payload?.financialCode);

    if (
      !agreementNumber ||
      !contractorAgencyName ||
      !resourceOrderNumber ||
      !incidentName ||
      !incidentNumber ||
      !financialCode
    ) {
      return { success: false, message: "Missing required fields", data: null };
    }

    const unitType = normalizeUnitType(payload?.unitType);
    const equipmentEntries = normalizeEquipmentEntries(
      payload?.equipmentEntries,
      unitType,
    );
    const personnelEntries = normalizePersonnelEntries(
      payload?.personnelEntries,
    );

    const contractorRepPrintedName = isBlank(payload?.contractorRepPrintedName)
      ? null
      : String(payload.contractorRepPrintedName).trim();

    const incidentSupervisorPosition = isBlank(payload?.incidentSupervisorPosition,) ? null : String(payload.incidentSupervisorPosition);

    const contractorRepSignature = isBlank(payload?.contractorRepSignature)
      ? null
      : String(payload.contractorRepSignature).trim();
    const incidentSupervisorPrintedName = isBlank(
      payload?.incidentSupervisorPrintedName,
    )
      ? null
      : String(payload.incidentSupervisorPrintedName).trim();
    const incidentSupervisorResourceOrderNumber = isBlank(
      payload?.incidentSupervisorResourceOrderNumber,
    )
      ? null
      : String(payload.incidentSupervisorResourceOrderNumber).trim();
    const incidentSupervisorSignature = isBlank(
      payload?.incidentSupervisorSignature,
    )
      ? null
      : String(payload.incidentSupervisorSignature).trim();

    const updateFields = {
      agreementNumber,
      contractorAgencyName,
      resourceOrderNumber,
      incidentName,
      incidentNumber,
      financialCode,
      equipmentMakeModel: payload.equipmentMakeModel || null,
      equipmentType: payload.equipmentType || null,
      serialVinNumber: payload.serialVinNumber || null,
      licenseIdNumber: payload.licenseIdNumber || null,
      transportRetained: normalizeTransportRetained(payload.transportRetained),
      isFirstLastTicket: !!payload.isFirstLastTicket,
      mobilization: !!payload.mobilization,
      demobilization: !!payload.demobilization,
      unitType,
      remarks: payload.remarks || null,

      contractorRepPrintedName,
      contractorRepSignature,
      incidentSupervisorPrintedName,
      incidentSupervisorPosition,
      incidentSupervisorResourceOrderNumber,
      incidentSupervisorSignature,

      updatedBy: user.userId || null,
    };

    await db.transaction(async (tx) => {
      await tx
        .update(form297Tickets)
        .set(updateFields)
        .where(eq(form297Tickets.id, id));
      await tx
        .delete(form297EquipmentEntries)
        .where(eq(form297EquipmentEntries.ticketId, id));
      await tx
        .delete(form297PersonnelEntries)
        .where(eq(form297PersonnelEntries.ticketId, id));

      if (equipmentEntries.length) {
        await tx
          .insert(form297EquipmentEntries)
          .values(equipmentEntries.map((e) => ({ ...e, ticketId: id })));
      }

      if (personnelEntries.length) {
        await tx
          .insert(form297PersonnelEntries)
          .values(personnelEntries.map((p) => ({ ...p, ticketId: id })));
      }
    });

    return {
      success: true,
      message: "Ticket updated successfully",
      data: null,
    };
  } catch (e) {
    return {
      success: false,
      message: e?.message || "Failed to update ticket",
      data: null,
    };
  }
};

const submitForm297TicketService = async (id, user) => {
  if (!id || !Number.isInteger(id)) {
    return {
      success: false,
      message: "Invalid or missing ticket ID",
      data: null,
    };
  }

  try {
    const existing = await db
      .select()
      .from(form297Tickets)
      .where(eq(form297Tickets.id, id))
      .limit(1);
    if (!existing || existing.length === 0) {
      return { success: false, message: "Ticket not found", data: null };
    }

    if (existing[0].status !== "draft") {
      return {
        success: false,
        message: "Only draft tickets can be submitted",
        data: null,
      };
    }

    const isOwner = existing?.[0]?.createdBy === user?.userId;
    const isAdmin = user?.isAdmin;

    if (!isOwner && !isAdmin) {
      return { success: false, message: "Forbidden", data: null };
    }

    await db
      .update(form297Tickets)
      .set({
        status: "submitted",
        submittedAt: new Date(),
        updatedBy: user.userId || null,
      })
      .where(eq(form297Tickets.id, id));

    return { success: true, message: "Ticket submitted", data: null };
  } catch (e) {
    return {
      success: false,
      message: e?.message || "Failed to submit ticket",
      data: null,
    };
  }
};

const completeForm297TicketService = async (id, user) => {
  const userId = user?.id;
  const isAdmin = user?.isAdmin;

  if (!isAdmin) {
    return {
      success: false,
      message: "Only user with admin or superadmin previliges can perform this action",
      data: null,
    };
  }

  if (!id || !Number.isInteger(id)) {
    return {
      success: false,
      message: "Invalid or missing ticket ID",
      data: null,
    };
  }

  try {
    const existing = await db
      .select()
      .from(form297Tickets)
      .where(eq(form297Tickets.id, id))
      .limit(1);
    if (!existing || existing.length === 0) {
      return { success: false, message: "Ticket not found", data: null };
    }

    if (existing[0].status !== "submitted") {
      return {
        success: false,
        message: "Only submitted tickets can be completed",
        data: null,
      };
    }

    await db
      .update(form297Tickets)
      .set({
        status: "completed",
        completedAt: new Date(),
        updatedBy: userId || null,
      })
      .where(eq(form297Tickets.id, id));

    return { success: true, message: "Ticket completed", data: null };
  } catch (e) {
    return {
      success: false,
      message: e?.message || "Failed to complete ticket",
      data: null,
    };
  }
};

// const supervisorSignForm297TicketService = async (id, payload, userId) => {
//   if (!id || !Number.isInteger(id)) {
//     return {
//       success: false,
//       message: "Invalid or missing ticket ID",
//       data: null,
//     };
//   }

//   try {
//     const existing = await db
//       .select()
//       .from(form297Tickets)
//       .where(eq(form297Tickets.id, id))
//       .limit(1);
//     if (!existing || existing.length === 0) {
//       return { success: false, message: "Ticket not found", data: null };
//     }

//     if (existing[0].status === "void") {
//       return { success: false, message: "Ticket is void", data: null };
//     }

//     if (existing[0].status === "completed") {
//       return {
//         success: false,
//         message: "Ticket is already completed",
//         data: null,
//       };
//     }

//     if (existing[0].status !== "contractor_signed") {
//       return {
//         success: false,
//         message: "Contractor signature required before supervisor signature",
//         data: null,
//       };
//     }

//     const incidentSupervisorPrintedName = toRequiredString(
//       payload?.incidentSupervisorPrintedName,
//     );
//     const incidentSupervisorResourceOrderNumber = toRequiredString(
//       payload?.incidentSupervisorResourceOrderNumber,
//     );
//     const incidentSupervisorSignature = toRequiredString(
//       payload?.incidentSupervisorSignature,
//     );
//     if (
//       !incidentSupervisorPrintedName ||
//       !incidentSupervisorResourceOrderNumber ||
//       !incidentSupervisorSignature
//     ) {
//       return {
//         success: false,
//         message:
//           "Supervisor printed name, resource order number, and signature are required",
//         data: null,
//       };
//     }

//     await db
//       .update(form297Tickets)
//       .set({
//         incidentSupervisorPrintedName,
//         incidentSupervisorResourceOrderNumber,
//         incidentSupervisorSignature,
//         completedAt: new Date(),
//         status: "completed",
//         updatedBy: userId || null,
//       })
//       .where(eq(form297Tickets.id, id));

//     return { success: true, message: "Supervisor signature saved", data: null };
//   } catch (e) {
//     return {
//       success: false,
//       message: e?.message || "Failed to sign",
//       data: null,
//     };
//   }
// };

const deleteForm297TicketService = async (id, userId, user) => {
  if (!id || !Number.isInteger(id)) {
    return {
      success: false,
      message: "Invalid or missing ticket ID",
      data: null,
    };
  }

  try {
    const existing = await db
      .select()
      .from(form297Tickets)
      .where(eq(form297Tickets.id, id))
      .limit(1);
    if (!existing || existing.length === 0) {
      return { success: false, message: "Ticket not found", data: null };
    }

    if (
      existing[0].status !== "draft" &&
      !user?.isAdmin
    ) {
      return {
        success: false,
        message: "Only draft tickets can be deleted",
        data: null,
      };
    }

    if (
      existing[0].createdBy &&
      userId &&
      existing[0].createdBy !== userId &&
      !user?.isAdmin
    ) {
      return { success: false, message: "Forbidden! Only for Admin", data: null };
    }

    await db.delete(form297Tickets).where(eq(form297Tickets.id, id));
    return { success: true, message: "Ticket deleted", data: null };
  } catch (e) {
    return {
      success: false,
      message: e?.message || "Failed to delete ticket",
      data: null,
    };
  }
};

// const getForm297TicketsByAgreementAndDateRangeService = async (agreementNumber, startDate, endDate, user) => {
//   const userId = user?.id;
//   const isSuperAdmin = !!user?.roleDetails?.isSuperAdmin;

//   if (!isSuperAdmin && !userId) {
//     return {
//       success: false,
//       message: "Unauthorized",
//       data: null,
//     };
//   }

//   if (!agreementNumber || !startDate || !endDate) {
//     return {
//       success: false,
//       message: "Agreement number, start date, and end date are required",
//       data: null,
//     };
//   }

//   try {
//     // Build base query with agreement number filter
//     let query = db
//       .select()
//       .from(form297Tickets)
//       .where(eq(form297Tickets.agreementNumber, agreementNumber));

//     // Add user filter if not super admin
//     if (!isSuperAdmin) {
//       query = query.where(eq(form297Tickets.createdBy, userId));
//     }


//     const tickets = await query;
//     console.log("agreement based numbers: " , tickets);


//     // Filter by date range manually (since we need to check equipment entries dates)
//     const results = [];
//     for (const ticket of tickets) {
//       // Get equipment entries to check dates
//       const equipmentEntries = await db
//         .select()
//         .from(form297EquipmentEntries)
//         .where(eq(form297EquipmentEntries.ticketId, ticket.id));

//       // Check if any equipment entry falls within the date range
//       const hasEntriesInRange = equipmentEntries.some(entry => {
//         if (!entry.workDate) return false;
//         const entryDate = new Date(entry.workDate);
//         const start = new Date(startDate);
//         const end = new Date(endDate);
//         return entryDate >= start && entryDate <= end;
//       });

//       if (hasEntriesInRange) {
//         // Get personnel entries for this ticket
//         const personnelEntries = await db
//           .select()
//           .from(form297PersonnelEntries)
//           .where(eq(form297PersonnelEntries.ticketId, ticket.id));

//         results.push({
//           ticket,
//           equipmentEntries,
//           personnelEntries,
//         });
//       }
//     }

//     return {
//       success: true,
//       message: "Tickets fetched successfully",
//       data: results,
//     };
//   } catch (e) {
//     return {
//       success: false,
//       message: e?.message || "Failed to fetch tickets",
//       data: null,
//     };
//   }
// };



const getForm297TicketsByAgreementAndDateRangeService = async (
  agreementNumber,
  startDate,
  endDate,
  user
) => {
  const userId = user?.id;
  const isAdmin = user?.isAdmin;

  console.log(isAdmin , userId);

  if (!isAdmin && !userId) {
    return { success: false, message: "Unauthorized", data: null };
  }

  console.log("passed")

  if (!agreementNumber || !startDate || !endDate) {
    return {
      success: false,
      message: "Agreement number, start date, and end date are required",
      data: null,
    };
  }

  try {


    const start = new Date(startDate);
    start.setHours(0, 0, 0, 0);

    const end = new Date(endDate);
    end.setHours(23, 59, 59, 999);

    const conditions = [
      eq(form297Tickets.agreementNumber, agreementNumber),
      gte(form297Tickets.createdAt, start),
      lte(form297Tickets.createdAt, end),
    ];


    // const conditions = [
    //   eq(form297Tickets.agreementNumber, agreementNumber),
    //   gte(form297Tickets.createdAt, startDate),
    //   lte(form297Tickets.createdAt, endDate),
    // ];

    if (!isAdmin) {
      conditions.push(eq(form297Tickets.createdBy, userId));
    }

    const results = await db
      .select({
        ticket: form297Tickets,
        equipmentEntry: form297EquipmentEntries,
        personnelEntry: form297PersonnelEntries,
      })
      .from(form297Tickets)
      .innerJoin(
        form297EquipmentEntries,
        eq(form297EquipmentEntries.ticketId, form297Tickets.id)
      )
      .leftJoin(
        form297PersonnelEntries,
        eq(form297PersonnelEntries.ticketId, form297Tickets.id)
      )
      .where(and(...conditions));

    // console.log("Resulses", results);

    return {
      success: true,
      message: "Tickets fetched successfully",
      data: results,
    };

  } catch (e) {
    return {
      success: false,
      message: e?.message || "Failed to fetch tickets",
      data: null,
    };
  }
};


module.exports = {
  getAllForm297TicketsService,
  getForm297TicketByIdService,
  createForm297TicketService,
  updateForm297TicketService,
  submitForm297TicketService,
  completeForm297TicketService,
  deleteForm297TicketService,
  getForm297TicketsByAgreementAndDateRangeService,
};
