const { eq, inArray } = require("drizzle-orm");

const { db } = require("../db/index");
const { allForms } = require("../db/schemas/allForms");
const {
  form288Reports,
  form288Employees,
  form288TimeEntries,
  form288CommissaryEntries,
} = require("../db/schemas/form288Schema");

const isBlank = (v) => v === null || v === undefined || String(v).trim() === "";

const toOptionalString = (v) => {
  if (isBlank(v)) return null;
  return String(v).trim();
};

const toNullableInt = (v) => {
  if (v === null || v === undefined || v === "") return null;
  const n = Number(v);
  return Number.isInteger(n) ? n : null;
};

const toNullableDecimal = (v) => {
  if (v === null || v === undefined || v === "") return null;
  const n = Number(v);
  if (Number.isNaN(n)) return null;
  return n;
};

const pick = (obj, keys) => {
  for (const k of keys) {
    if (!obj) continue;
    const v = obj[k];
    if (!isBlank(v)) return v;
  }
  return null;
};

const normalizeWorkDate = (workDate, rowIndex) => {
  if (isBlank(workDate)) return null;
  const s = String(workDate).trim();

  if (/^\d{4}-\d{2}-\d{2}$/.test(s)) return s;

  const d = new Date(s);
  if (!Number.isNaN(d?.getTime?.())) {
    return d.toISOString().slice(0, 10);
  }

  return null;
};

const normalizeReport = (payload) => {
  const p = payload || {};

  return {
    hiredAt: toOptionalString(pick(p, ["hiredAt", "box1"])),
    employeeCommonIdentifier: toOptionalString(pick(p, ["employeeCommonIdentifier", "box2"])),
    typeOfEmployment: toOptionalString(pick(p, ["typeOfEmployment", "box3"])),
    hiringUnitName: toOptionalString(pick(p, ["hiringUnitName", "box4"])),

    hiringUnitPhoneNumber: toOptionalString(pick(p, ["hiringUnitPhoneNumber", "box6"])),
    hiringUnitFaxNumber: toOptionalString(pick(p, ["hiringUnitFaxNumber", "box7"])),

    incidentName: toOptionalString(pick(p, ["incidentName", "box8"])),
    incidentOrderNumber: toOptionalString(pick(p, ["incidentOrderNumber", "box9"])),
    fireCode: toOptionalString(pick(p, ["fireCode", "box10"])),
    resourceRequestNumber: toOptionalString(pick(p, ["resourceRequestNumber", "box11"])),
    remarks: toOptionalString(pick(p, ["remarks", "box19"])),

    employeeSignature: toOptionalString(pick(p, ["employeeSignature", "box20Signature"])),
    timeOfficerSignature: toOptionalString(pick(p, ["timeOfficerSignature", "box21Signature"])),
  };
};

const normalizeEmployees = (employees) => {
  const list = Array.isArray(employees) ? employees : [];

  return list
    .map((e) => ({
      employeeName: toOptionalString(pick(e, ["employeeName", "name", "box5"])),
      employeeUserId: toNullableInt(e?.employeeUserId),
      name: toOptionalString(pick(e, ["name", "box5"])),
      positionCode: toOptionalString(pick(e, ["positionCode", "box12"])),
      adClass: toOptionalString(pick(e, ["adClass", "box13"])),
      adRate: toOptionalString(pick(e, ["adRate", "box14"])),
      homeHiringUnitAccountingCode: toOptionalString(pick(e, ["homeHiringUnitAccountingCode", "box15"])),
      timeEntries: Array.isArray(e?.timeEntries) ? e.timeEntries : [],
    }))
    .filter((e) => {
      const hasAny =
        !isBlank(e.employeeName) ||
        e.employeeUserId !== null ||
        !isBlank(e.name) ||
        !isBlank(e.positionCode) ||
        !isBlank(e.adClass) ||
        !isBlank(e.adRate) ||
        !isBlank(e.homeHiringUnitAccountingCode) ||
        (Array.isArray(e.timeEntries) && e.timeEntries.length > 0);
      return hasAny;
    });
};

const normalizeTimeEntries = (entries) => {
  const list = Array.isArray(entries) ? entries : [];

  const trimmed = list.slice(0, 21);

  return trimmed
    .map((t, idx) => ({
      workDate: normalizeWorkDate(t?.workDate, toNullableInt(t?.rowIndex) ?? idx),
      startTime: t?.startTime || null,
      stopTime: t?.stopTime || null,
      hours: toNullableDecimal(t?.hours),
      rowIndex: toNullableInt(t?.rowIndex) ?? idx,
    }))
    .filter((t) => !!t.workDate);
};

const normalizeCommissary = (entries) => {
  const list = Array.isArray(entries) ? entries : [];

  return list
    .map((c) => ({
      month: toOptionalString(c?.month),
      day: toOptionalString(c?.day),
      category: toOptionalString(c?.category),
      reimbursement: toNullableDecimal(c?.reimbursement),
      deduction: toNullableDecimal(c?.deduction),
      fireCode: toOptionalString(c?.fireCode),
    }))
    .filter((c) => {
      return (
        !isBlank(c.month) ||
        !isBlank(c.day) ||
        !isBlank(c.category) ||
        c.reimbursement !== null ||
        c.deduction !== null ||
        !isBlank(c.fireCode)
      );
    });
};

const canSeeAll = (user) => user?.isAdmin;

const getAllForm288ReportsService = async (user) => {
  try {
    const userId = user?.id;
    const allowAll = user?.isAdmin;

    if (!allowAll && !userId) {
      return { success: false, message: "Unauthorized", data: null };
    }

    let query = db.select().from(form288Reports);
    if (!allowAll) {
      query = query.where(eq(form288Reports.createdBy, userId));
    }

    const rows = await query.orderBy(form288Reports.id);

    return { success: true, message: "Form 288 reports fetched", data: rows };
  } catch {
    return { success: false, message: "Failed to fetch Form 288 reports", data: null };
  }
};

const getForm288ReportByIdService = async (id, user) => {
  if (!id || !Number.isInteger(id)) {
    return { success: false, message: "Invalid or missing report ID", data: null };
  }

  try {
    const userId = user?.id;
    const allowAll = user?.isAdmin;

    if (!allowAll && !userId) {
      return { success: false, message: "Unauthorized", data: null };
    }

    const reportRows = await db
      .select()
      .from(form288Reports)
      .where(eq(form288Reports.id, id))
      .limit(1);

    if (!reportRows || reportRows.length === 0) {
      return { success: false, message: "Report not found", data: null };
    }

    const report = reportRows[0];

    if (!allowAll && report?.createdBy !== userId) {
      return { success: false, message: "Forbidden", data: null };
    }

    const employees = await db
      .select()
      .from(form288Employees)
      .where(eq(form288Employees.reportId, id));

    const commissaryEntries = await db
      .select()
      .from(form288CommissaryEntries)
      .where(eq(form288CommissaryEntries.reportId, id));

    const employeeIds = employees.map((e) => e.id);
    const timeEntries = employeeIds.length
      ? await db
          .select()
          .from(form288TimeEntries)
          .where(inArray(form288TimeEntries.employeeId, employeeIds))
      : [];

    const timeEntriesByEmployeeId = {};
    timeEntries.forEach((t) => {
      const key = String(t.employeeId);
      if (!timeEntriesByEmployeeId[key]) timeEntriesByEmployeeId[key] = [];
      timeEntriesByEmployeeId[key].push(t);
    });

    return {
      success: true,
      message: "Report fetched successfully",
      data: {
        report,
        employees,
        commissaryEntries,
        timeEntriesByEmployeeId,
      },
    };
  } catch (e) {
    return { success: false, message: e?.message || "Failed to fetch report", data: null };
  }
};

const createForm288ReportService = async (payload, user) => {
  try {
    const userId = user?.id || null;

    const reportData = normalizeReport(payload);
    const employees = normalizeEmployees(payload?.employees);
    const commissaryEntries = normalizeCommissary(payload?.commissaryEntries);

    const result = await db.transaction(async (tx) => {
      let [form] = await tx
        .select()
        .from(allForms)
        .where(eq(allForms.formName, "form288"))
        .limit(1);

      if (!form) {
        const inserted = await tx.insert(allForms).values({
          formName: "form288",
          formDescription: "Form 288",
        });

        const insertId = inserted?.[0]?.insertId;
        form = { id: insertId };
      }

      const ids = await tx
        .insert(form288Reports)
        .values({
          ...reportData,
          formId: form?.id || null,
          createdBy: userId,
          updatedBy: userId,
        })
        .$returningId();

      const reportId = ids?.[0]?.id;
      if (!reportId) return { reportId: null };

      if (commissaryEntries.length) {
        await tx
          .insert(form288CommissaryEntries)
          .values(commissaryEntries.map((c) => ({ ...c, reportId })));
      }

      for (const emp of employees) {
        const empIds = await tx
          .insert(form288Employees)
          .values({
            reportId,
            employeeName: emp.employeeName,
            employeeUserId: emp.employeeUserId,
            name: emp.name,
            positionCode: emp.positionCode,
            adClass: emp.adClass,
            adRate: emp.adRate,
            homeHiringUnitAccountingCode: emp.homeHiringUnitAccountingCode,
          })
          .$returningId();

        const employeeId = empIds?.[0]?.id;
        if (!employeeId) continue;

        const times = normalizeTimeEntries(emp.timeEntries);
        if (times.length) {
          await tx
            .insert(form288TimeEntries)
            .values(times.map((t) => ({ ...t, employeeId })));
        }
      }

      return { reportId };
    });

    if (!result.reportId) {
      return { success: false, message: "Failed to create report", data: null };
    }

    return {
      success: true,
      message: "Report created successfully",
      data: { id: result.reportId },
    };
  } catch (e) {
    return { success: false, message: e?.message || "Failed to create report", data: null };
  }
};

const updateForm288ReportService = async (payload, user) => {
  const id = payload?.id;
  if (!id || !Number.isInteger(id)) {
    return { success: false, message: "Invalid or missing report ID", data: null };
  }

  try {
    const userId = user?.id || null;

    const existing = await db
      .select()
      .from(form288Reports)
      .where(eq(form288Reports.id, id))
      .limit(1);

    if (!existing || existing.length === 0) {
      return { success: false, message: "Report not found", data: null };
    }

    if (existing[0].status !== "draft") {
      return { success: false, message: "Only draft reports can be updated", data: null };
    }

    if (existing[0].createdBy && user?.id && existing[0].createdBy !== user.id && !user?.isAdmin) {
      return { success: false, message: "Forbidden", data: null };
    }

    const reportData = normalizeReport(payload);
    const employees = normalizeEmployees(payload?.employees);
    const commissaryEntries = normalizeCommissary(payload?.commissaryEntries);

    await db.transaction(async (tx) => {
      await tx
        .update(form288Reports)
        .set({
          ...reportData,
          updatedBy: userId,
        })
        .where(eq(form288Reports.id, id));

      await tx
        .delete(form288CommissaryEntries)
        .where(eq(form288CommissaryEntries.reportId, id));

      if (commissaryEntries.length) {
        await tx
          .insert(form288CommissaryEntries)
          .values(commissaryEntries.map((c) => ({ ...c, reportId: id })));
      }

      const oldEmployees = await tx
        .select({ id: form288Employees.id })
        .from(form288Employees)
        .where(eq(form288Employees.reportId, id));

      for (const emp of oldEmployees) {
        await tx.delete(form288TimeEntries).where(eq(form288TimeEntries.employeeId, emp.id));
      }

      await tx.delete(form288Employees).where(eq(form288Employees.reportId, id));

      for (const emp of employees) {
        const empIds = await tx
          .insert(form288Employees)
          .values({
            reportId: id,
            employeeName: emp.employeeName,
            employeeUserId: emp.employeeUserId,
            name: emp.name,
            positionCode: emp.positionCode,
            adClass: emp.adClass,
            adRate: emp.adRate,
            homeHiringUnitAccountingCode: emp.homeHiringUnitAccountingCode,
          })
          .$returningId();

        const employeeId = empIds?.[0]?.id;
        if (!employeeId) continue;

        const times = normalizeTimeEntries(emp.timeEntries);
        if (times.length) {
          await tx
            .insert(form288TimeEntries)
            .values(times.map((t) => ({ ...t, employeeId })));
        }
      }
    });

    return { success: true, message: "Report updated successfully", data: null };
  } catch (e) {
    return { success: false, message: e?.message || "Failed to update report", data: null };
  }
};

const submitForm288ReportService = async (id, user) => {
  if (!id || !Number.isInteger(id)) {
    return { success: false, message: "Invalid or missing report ID", data: null };
  }

  try {
    const existing = await db
      .select()
      .from(form288Reports)
      .where(eq(form288Reports.id, id))
      .limit(1);

    if (!existing || existing.length === 0) {
      return { success: false, message: "Report not found", data: null };
    }

    if (existing[0].status !== "draft") {
      return { success: false, message: "Only draft reports can be submitted", data: null };
    }

    if (existing[0].createdBy && user?.id && existing[0].createdBy !== user.id && !user?.isAdmin) {
      return { success: false, message: "Forbidden", data: null };
    }

    await db
      .update(form288Reports)
      .set({
        status: "submitted",
        submittedAt: new Date(),
        updatedBy: user?.id || null,
      })
      .where(eq(form288Reports.id, id));

    return { success: true, message: "Report submitted", data: null };
  } catch (e) {
    return { success: false, message: e?.message || "Failed to submit report", data: null };
  }
};

const completeForm288ReportService = async (id, user) => {
  if (!user?.isAdmin) {
    return { success: false, message: "Forbidden! Only admin can do", data: null };
  }

  if (!id || !Number.isInteger(id)) {
    return { success: false, message: "Invalid or missing report ID", data: null };
  }

  try {
    const existing = await db
      .select()
      .from(form288Reports)
      .where(eq(form288Reports.id, id))
      .limit(1);

    if (!existing || existing.length === 0) {
      return { success: false, message: "Report not found", data: null };
    }

    if (existing[0].status !== "submitted") {
      return { success: false, message: "Only submitted reports can be completed", data: null };
    }

    await db
      .update(form288Reports)
      .set({
        status: "completed",
        completedAt: new Date(),
        updatedBy: user?.id || null,
      })
      .where(eq(form288Reports.id, id));

    return { success: true, message: "Report completed", data: null };
  } catch (e) {
    return { success: false, message: e?.message || "Failed to complete report", data: null };
  }
};

const deleteForm288ReportService = async (id, user) => {
  if (!id || !Number.isInteger(id)) {
    return { success: false, message: "Invalid or missing report ID", data: null };
  }

  try {
    const existing = await db
      .select()
      .from(form288Reports)
      .where(eq(form288Reports.id, id))
      .limit(1);

    if (!existing || existing.length === 0) {
      return { success: false, message: "Report not found", data: null };
    }

    const canDelete = existing[0].status === "draft" || user?.isAdmin;
    if (!canDelete) {
      return { success: false, message: "Only draft reports can be deleted", data: null };
    }

    if (existing[0].createdBy && user?.id && existing[0].createdBy !== user.id && !user?.isAdmin) {
      return { success: false, message: "Forbidden", data: null };
    }

    await db.delete(form288Reports).where(eq(form288Reports.id, id));

    return { success: true, message: "Report deleted", data: null };
  } catch (e) {
    return { success: false, message: e?.message || "Failed to delete report", data: null };
  }
};

module.exports = {
  getAllForm288ReportsService,
  getForm288ReportByIdService,
  createForm288ReportService,
  updateForm288ReportService,
  submitForm288ReportService,
  completeForm288ReportService,
  deleteForm288ReportService,
};
