const { eq, desc } = require("drizzle-orm");

const { db } = require("../db/index");
const { allForms } = require("../db/schemas/allForms");
const {
  form268Reports,
  form268RateEntries,
} = require("../db/schemas/form268Schema");
const { users } = require("../db/schemas/userSchema");

const isBlank = (v) => v === null || v === undefined || String(v).trim() === "";

const toOptionalString = (v) => {
  if (isBlank(v)) return null;
  return String(v).trim();
};

const toNullableInt = (v) => {
  if (v === null || v === undefined || v === "") return null;
  const n = Number(v);
  return Number.isInteger(n) ? n : null;
};

const toNullableDecimal = (v) => {
  if (v === null || v === undefined || v === "") return null;
  const n = Number(v);
  if (Number.isNaN(n)) return null;
  return n;
};

const toNullableDate = (v) => {
  if (isBlank(v)) return null;
  const s = String(v).trim();
  if (/^\d{4}-\d{2}-\d{2}$/.test(s)) return s;
  const d = new Date(s);
  if (!Number.isNaN(d?.getTime?.())) {
    return d.toISOString().slice(0, 10);
  }
  return null;
};

const toNullableTime = (v) => {
  if (isBlank(v)) return null;
  const s = String(v).trim();
  if (/^\d{2}:\d{2}(:\d{2})?$/.test(s)) return s;
  return null;
};

const pick = (obj, keys) => {
  for (const k of keys) {
    if (!obj) continue;
    const v = obj[k];
    if (!isBlank(v)) return v;
  }
  return null;
};

const normalizeReport = (payload) => {
  const p = payload || {};

  return {
    // Box 1: Contractor
    contractorName: toOptionalString(pick(p, ["contractorName", "box1Name"])),
    contractorAddress: toOptionalString(pick(p, ["contractorAddress", "box1Address"])),

    // Box 2: Incident Name
    incidentName: toOptionalString(pick(p, ["incidentName", "box2"])),

    // Box 3: Agreement Number
    agreementNumber: toOptionalString(pick(p, ["agreementNumber", "box3"])),

    // Box 4: Agreement Dates
    agreementBeginDate: toNullableDate(pick(p, ["agreementBeginDate", "box4Begin"])),
    agreementEndDate: toNullableDate(pick(p, ["agreementEndDate", "box4End"])),

    // Box 5: Equipment
    equipmentMake: toOptionalString(pick(p, ["equipmentMake", "box5Make"])),
    equipmentModel: toOptionalString(pick(p, ["equipmentModel", "box5Model"])),
    equipmentSerialNumber: toOptionalString(pick(p, ["equipmentSerialNumber", "box5Serial"])),
    equipmentDescription: toOptionalString(pick(p, ["equipmentDescription", "box5Description"])),

    // Box 6: Point of Hire
    pointOfHire: toOptionalString(pick(p, ["pointOfHire", "box6"])),

    // Box 7: Date of Hire
    dateOfHire: toNullableDate(pick(p, ["dateOfHire", "box7"])),

    // Box 8: Time of Hire
    timeOfHire: toNullableTime(pick(p, ["timeOfHire", "box8"])),

    // Box 9: Admin Office for Payment
    adminOfficeForPayment: toOptionalString(pick(p, ["adminOfficeForPayment", "box9"])),

    // Box 10: Operating Supplies By
    operatingSuppliesBy: toOptionalString(pick(p, ["operatingSuppliesBy", "box10"])),

    // Box 11: Operator Furnished By
    operatorFurnishedBy: toOptionalString(pick(p, ["operatorFurnishedBy", "box11"])),

    // Box 12: Resource Order Number
    resourceOrderNumber: toOptionalString(pick(p, ["resourceOrderNumber", "box12"])),

    // Box 17: Guarantee
    guaranteeAmount: toNullableDecimal(pick(p, ["guaranteeAmount", "box17"])),

    // Box 19: Charge Code
    chargeCode: toOptionalString(pick(p, ["chargeCode", "box19"])),

    // Box 20: Object Code
    objectCode: toOptionalString(pick(p, ["objectCode", "box20"])),

    // Box 21: Equipment Released/Withdrawn
    equipmentReleased: toOptionalString(pick(p, ["equipmentReleased", "box21Status"])),
    equipmentReleasedDate: toNullableDate(pick(p, ["equipmentReleasedDate", "box21Date"])),
    equipmentReleasedTime: toNullableTime(pick(p, ["equipmentReleasedTime", "box21Time"])),

    // Box 22: Remarks
    remarks: toOptionalString(pick(p, ["remarks", "box22"])),

    // Financial Summary
    totalAmountEarned: toNullableDecimal(pick(p, ["totalAmountEarned", "box16"])),
    amountDue: toNullableDecimal(pick(p, ["amountDue", "box18"])),
    grossAmountDue: toNullableDecimal(pick(p, ["grossAmountDue", "box23"])),
    previousPageAmount: toNullableDecimal(pick(p, ["previousPageAmount", "box24"])),
    totalAmountDue: toNullableDecimal(pick(p, ["totalAmountDue", "box25"])),
    deductions: toNullableDecimal(pick(p, ["deductions", "box26"])),
    additions: toNullableDecimal(pick(p, ["additions", "box27Additions"])),
    gratuity: toNullableDecimal(pick(p, ["gratuity", "box27Gratuity"])),
    netAmountDue: toNullableDecimal(pick(p, ["netAmountDue", "box28"])),

    // Box 30: Contractor's Signature
    contractorSignature: toOptionalString(pick(p, ["contractorSignature", "box30"])),

    // Box 31: Contractor Date
    contractorSignatureDate: toNullableDate(pick(p, ["contractorSignatureDate", "box31"])),

    // Box 32: Receiving Officer's Signature
    receivingOfficerSignature: toOptionalString(pick(p, ["receivingOfficerSignature", "box32"])),

    // Box 33: Receiving Officer Date
    receivingOfficerDate: toNullableDate(pick(p, ["receivingOfficerDate", "box33"])),

    // Box 34: Contractor Print Name and Title
    contractorPrintName: toOptionalString(pick(p, ["contractorPrintName", "box34Name"])),
    contractorTitle: toOptionalString(pick(p, ["contractorTitle", "box34Title"])),

    // Box 35: Receiving Officer Print Name and Title
    receivingOfficerPrintName: toOptionalString(pick(p, ["receivingOfficerPrintName", "box35Name"])),
    receivingOfficerTitle: toOptionalString(pick(p, ["receivingOfficerTitle", "box35Title"])),
  };
};

const normalizeRateEntries = (entries) => {
  const list = Array.isArray(entries) ? entries : [];
  console.log("[Form268 Service] normalizeRateEntries input:", list);

  const result = list
    .map((e, idx) => {
      const year = toOptionalString(pick(e, ["entryYear", "year", "box13Year"]));
      const month = toOptionalString(pick(e, ["entryMonth", "month", "box13Month"]));
      const day = toOptionalString(pick(e, ["entryDay", "day", "box13Day"]));

      // Validate date format - reject placeholder text
      const validYear = /^\d{4}$/.test(year) ? year : null;
      const validMonth = /^\d{1,2}$/.test(month) ? month : null;
      const validDay = /^\d{1,2}$/.test(day) ? day : null;

      return {
        entryYear: validYear,
        entryMonth: validMonth,
        entryDay: validDay,

        // Box 14: Work or Daily Rate
        workUnits: toOptionalString(pick(e, ["workUnits", "box14aUnits"])),
        workRate: toOptionalString(pick(e, ["workRate", "box14bRate"])),
        workAmount: toOptionalString(pick(e, ["workAmount", "box14cAmount"])),

        // Box 15: Special Rate
        specialUnits: toOptionalString(pick(e, ["specialUnits", "box15aUnits"])),
        specialRate: toOptionalString(pick(e, ["specialRate", "box15bRate"])),
        specialAmount: toOptionalString(pick(e, ["specialAmount", "box15cAmount"])),

        // Boxes 16-18 (row-level)
        totalAmountEarned: toOptionalString(pick(e, ["totalAmountEarned", "box16"])),
        guaranteeAmount: toOptionalString(pick(e, ["guaranteeAmount", "box17"])),
        amount: toOptionalString(pick(e, ["amount", "box18"])),

        rowIndex: toNullableInt(e?.rowIndex) ?? idx,
      };
    })
    .filter((e) => {
      const hasAny =
        !isBlank(e.entryYear) ||
        !isBlank(e.entryMonth) ||
        !isBlank(e.entryDay) ||
        !isBlank(e.workUnits) ||
        !isBlank(e.workRate) ||
        !isBlank(e.workAmount) ||
        !isBlank(e.specialUnits) ||
        !isBlank(e.specialRate) ||
        !isBlank(e.specialAmount) ||
        !isBlank(e.totalAmountEarned) ||
        !isBlank(e.guaranteeAmount) ||
        !isBlank(e.amount);
      return hasAny;
    });

  console.log("[Form268 Service] normalizeRateEntries output:", result);
  return result;
};

const canSeeAll = (user) => user?.isAdmin;

const getAllForm268ReportsService = async (user) => {
  try {
    const userId = user?.id;
    const allowAll = user?.isAdmin;

    if (!allowAll && !userId) {
      return { success: false, message: "Unauthorized", data: null };
    }

    let query = db.select().from(form268Reports).leftJoin(users , eq(form268Reports.createdBy , users.id));
    if (!allowAll) {
      query = query.where(eq(form268Reports.createdBy, userId));
    }


    // const rows = await query.orderBy(form268Reports.id);
    const rows = await query.orderBy(desc(form268Reports.updatedAt));

    // console.log("rows" , rows);



    return { success: true, message: "Form 268 reports fetched", data: rows };
  } catch {
    return { success: false, message: "Failed to fetch Form 268 reports", data: null };
  }
};

const getForm268ReportByIdService = async (id, user) => {
  if (!id || !Number.isInteger(id)) {
    return { success: false, message: "Invalid or missing report ID", data: null };
  }

  try {
    const userId = user?.id;
    const allowAll = user?.isAdmin;

    if (!allowAll && !userId) {
      return { success: false, message: "Unauthorized", data: null };
    }

    const reportRows = await db
      .select()
      .from(form268Reports)
      .where(eq(form268Reports.id, id))
      .limit(1);

    if (!reportRows || reportRows.length === 0) {
      return { success: false, message: "Report not found", data: null };
    }

    const report = reportRows[0];

    if (!allowAll && report?.createdBy !== userId) {
      return { success: false, message: "Forbidden", data: null };
    }

    const rateEntries = await db
      .select()
      .from(form268RateEntries)
      .where(eq(form268RateEntries.reportId, id))
      .orderBy(form268RateEntries.rowIndex);

    // console.log("[Form268 Service] getForm268ReportById rateEntries from DB:", rateEntries);

    return {
      success: true,
      message: "Report fetched successfully",
      data: {
        report,
        rateEntries,
      },
    };
  } catch (e) {
    return { success: false, message: e?.message || "Failed to fetch report", data: null };
  }
};

const createForm268ReportService = async (payload, user) => {
  try {
    const userId = user?.id || null;

    const reportData = normalizeReport(payload);
    const rateEntries = normalizeRateEntries(payload?.rateEntries);

    const result = await db.transaction(async (tx) => {
      let [form] = await tx
        .select()
        .from(allForms)
        .where(eq(allForms.formName, "form268"))
        .limit(1);

      if (!form) {
        const inserted = await tx.insert(allForms).values({
          formName: "form268",
          formDescription: "Emergency Equipment Use Invoice (OF-286)",
        });

        const insertId = inserted?.[0]?.insertId;
        form = { id: insertId };
      }

      const ids = await tx
        .insert(form268Reports)
        .values({
          ...reportData,
          formId: form?.id || null,
          createdBy: userId,
          updatedBy: userId,
        })
        .$returningId();

      const reportId = ids?.[0]?.id;
      if (!reportId) return { reportId: null };

      if (rateEntries.length) {
        await tx
          .insert(form268RateEntries)
          .values(rateEntries.map((r) => ({ ...r, reportId })));
      }

      return { reportId };
    });

    if (!result.reportId) {
      return { success: false, message: "Failed to create report", data: null };
    }

    return {
      success: true,
      message: "Report created successfully",
      data: { id: result.reportId },
    };
  } catch (e) {
    return { success: false, message: e?.message || "Failed to create report", data: null };
  }
};

const updateForm268ReportService = async (payload, user) => {
  const id = payload?.id;
  if (!id || !Number.isInteger(id)) {
    return { success: false, message: "Invalid or missing report ID", data: null };
  }

  try {
    const userId = user?.id || null;

    console.log("[Form268 Service] update payload:", JSON.stringify(payload, null, 2));
    console.log("[Form268 Service] rateEntries from payload:", payload?.rateEntries);

    const existing = await db
      .select()
      .from(form268Reports)
      .where(eq(form268Reports.id, id))
      .limit(1);

    if (!existing || existing.length === 0) {
      return { success: false, message: "Report not found", data: null };
    }

    if (existing[0].status !== "draft") {
      return { success: false, message: "Only draft reports can be updated", data: null };
    }

    if (existing[0].createdBy && user?.id && existing[0].createdBy !== user.id && !user?.isAdmin) {
      return { success: false, message: "Forbidden", data: null };
    }

    const reportData = normalizeReport(payload);
    const rateEntries = normalizeRateEntries(payload?.rateEntries);

    await db.transaction(async (tx) => {
      await tx
        .update(form268Reports)
        .set({
          ...reportData,
          updatedBy: userId,
        })
        .where(eq(form268Reports.id, id));

      await tx.delete(form268RateEntries).where(eq(form268RateEntries.reportId, id));

      console.log("[Form268 Service] inserting rateEntries:", rateEntries);
      if (rateEntries.length) {
        const insertResult = await tx
          .insert(form268RateEntries)
          .values(rateEntries.map((r) => ({ ...r, reportId: id })));
        console.log("[Form268 Service] insert result:", insertResult);
      }
    });

    return { success: true, message: "Report updated successfully", data: null };
  } catch (e) {
    return { success: false, message: e?.message || "Failed to update report", data: null };
  }
};

const submitForm268ReportService = async (id, user) => {
  if (!id || !Number.isInteger(id)) {
    return { success: false, message: "Invalid or missing report ID", data: null };
  }

  try {
    const existing = await db
      .select()
      .from(form268Reports)
      .where(eq(form268Reports.id, id))
      .limit(1);

    if (!existing || existing.length === 0) {
      return { success: false, message: "Report not found", data: null };
    }

    if (existing[0].status !== "draft") {
      return { success: false, message: "Only draft reports can be submitted", data: null };
    }

    if (existing[0].createdBy && user?.id && existing[0].createdBy !== user.id && !user?.isAdmin) {
      return { success: false, message: "Forbidden", data: null };
    }

    await db
      .update(form268Reports)
      .set({
        status: "submitted",
        submittedAt: new Date(),
        updatedBy: user?.id || null,
      })
      .where(eq(form268Reports.id, id));

    return { success: true, message: "Report submitted", data: null };
  } catch (e) {
    return { success: false, message: e?.message || "Failed to submit report", data: null };
  }
};

const completeForm268ReportService = async (id, user) => {
  if (!user?.isAdmin) {
    return { success: false, message: "Forbidden! Only admin can do", data: null };
  }

  if (!id || !Number.isInteger(id)) {
    return { success: false, message: "Invalid or missing report ID", data: null };
  }

  try {
    const existing = await db
      .select()
      .from(form268Reports)
      .where(eq(form268Reports.id, id))
      .limit(1);

    if (!existing || existing.length === 0) {
      return { success: false, message: "Report not found", data: null };
    }

    if (existing[0].status !== "submitted") {
      return { success: false, message: "Only submitted reports can be completed", data: null };
    }

    await db
      .update(form268Reports)
      .set({
        status: "completed",
        completedAt: new Date(),
        updatedBy: user?.id || null,
      })
      .where(eq(form268Reports.id, id));

    return { success: true, message: "Report completed", data: null };
  } catch (e) {
    return { success: false, message: e?.message || "Failed to complete report", data: null };
  }
};

const deleteForm268ReportService = async (id, user) => {
  if (!id || !Number.isInteger(id)) {
    return { success: false, message: "Invalid or missing report ID", data: null };
  }

  try {
    const existing = await db
      .select()
      .from(form268Reports)
      .where(eq(form268Reports.id, id))
      .limit(1);

    if (!existing || existing.length === 0) {
      return { success: false, message: "Report not found", data: null };
    }

    const canDelete = existing[0].status === "draft" || user?.isAdmin;
    if (!canDelete) {
      return { success: false, message: "Only draft reports can be deleted", data: null };
    }

    if (existing[0].createdBy && user?.id && existing[0].createdBy !== user.id && !user?.isAdmin) {
      return { success: false, message: "Forbidden", data: null };
    }

    await db.delete(form268Reports).where(eq(form268Reports.id, id));

    return { success: true, message: "Report deleted", data: null };
  } catch (e) {
    return { success: false, message: e?.message || "Failed to delete report", data: null };
  }
};

module.exports = {
  getAllForm268ReportsService,
  getForm268ReportByIdService,
  createForm268ReportService,
  updateForm268ReportService,
  submitForm268ReportService,
  completeForm268ReportService,
  deleteForm268ReportService,
};
