const { eq, gte, lte, and, desc } = require("drizzle-orm");

const { db } = require("../db/index");
const { allForms } = require("../db/schemas/allForms");
const { form261Reports, form261TimeEntries } = require("../db/schemas/form261Schema");
const dayjs = require("dayjs");

const isBlank = (v) => v === null || v === undefined || String(v).trim() === "";

const toRequiredString = (v) => {
  if (isBlank(v)) return null;
  return String(v).trim();
};

const toOptionalString = (v) => {
  if (isBlank(v)) return null;
  return String(v).trim();
};

const toNullableInt = (v) => {
  if (v === null || v === undefined || v === "") return null;
  const n = Number(v);
  return Number.isInteger(n) ? n : null;
};

const normalizeDayIndex = (v) => {
  const n = toNullableInt(v);
  if (!n) return null;
  if (n < 1 || n > 5) return null;
  return n;
};

const normalizeTimeEntries = (entries) => {
  const list = Array.isArray(entries) ? entries : [];

  return list
    .filter((e) => e && normalizeDayIndex(e.dayIndex))
    .map((e) => {
      const dayIndex = normalizeDayIndex(e.dayIndex);
      const rowIndex = toNullableInt(e.rowIndex) ?? 0;

      return {
        remarksNo: toNullableInt(e.remarksNo),
        employeeName: toOptionalString(e.employeeName),
        employeeUserId: toNullableInt(e.employeeUserId),
        classification: toOptionalString(e.classification),
        rowIndex,
        dayIndex,
        onTime: e.onTime || null,
        offTime: e.offTime || null,
      };
    })
    .filter((e) => {
      return (
        !isBlank(e.employeeName) ||
        e.employeeUserId !== null ||
        !isBlank(e.classification) ||
        !isBlank(e.onTime) ||
        !isBlank(e.offTime) ||
        e.remarksNo !== null
      );
    });
};

const getRoleName = (user) => String(user?.roleDetails?.name || "").toLowerCase();
const canSeeAll = (user) => user?.isAdmin;

const getAllForm261ReportsService = async (user) => {
  try {
    const userId = user?.id;
    const canSeeAll = user?.isAdmin;

    if (!canSeeAll && !userId) {
      return { success: false, message: "Unauthorized", data: null };
    }

    let query = db.select().from(form261Reports);
    if (!canSeeAll) {
      query = query.where(eq(form261Reports.createdBy, userId));
    }

    const rows = await query.orderBy(form261Reports.id);

    return {
      success: true,
      message: "Crew time reports fetched successfully",
      data: rows,
    };
  } catch {
    return {
      success: false,
      message: "Failed to fetch crew time reports",
      data: null,
    };
  }
};

const getForm261ReportByIdService = async (id, user) => {
  if (!id || !Number.isInteger(id)) {
    return { success: false, message: "Invalid or missing report ID", data: null };
  }

  try {
    const userId = user?.id;
    const canSeeAll = user?.isAdmin;

    if (!canSeeAll && !userId) {
      return { success: false, message: "Unauthorized", data: null };
    }

    const report = await db
      .select()
      .from(form261Reports)
      .where(eq(form261Reports.id, id))
      .limit(1);

    if (!report || report.length === 0) {
      return { success: false, message: "Report not found", data: null };
    }

    const row = report[0];

    console.log(row);

    if (!canSeeAll && row?.createdBy !== userId) {
      return { success: false, message: "Forbidden", data: null };
    }

    const timeEntries = await db
      .select()
      .from(form261TimeEntries)
      .where(eq(form261TimeEntries.reportId, id));

    return {
      success: true,
      message: "Report fetched successfully",
      data: { report: row, timeEntries },
    };
  } catch {
    return { success: false, message: "Failed to fetch report", data: null };
  }
};

const createForm261ReportService = async (payload, user) => {
  // console.log("createForm261ReportService payload:", payload);
  // if (!canAdminCreate(user)) {
  //   return { success: false, message: "Forbidden", data: null };
  // }

  // look for this form if already created for the day, don't allow user to create again for that day ok? also to admin or superadmin as well..no useris allowed to create 

  let existing;
  try{
  existing = await db.select().from(form261Reports).where(eq(form261Reports.createdBy , user.id)).orderBy(desc(form261Reports.createdAt)).limit(1);
  console.log(existing[0]);

  }
  catch(e)
  {
    console.log(e.message)
  }


  if(existing[0])
  {
    const created = dayjs(existing.createdAt).utc();
        const today = dayjs().utc();
        const isToday = created.isSame(today, 'day');
    
    
        console.log("created", created, today, isToday)
    
        if (isToday) {
    
          return {
            success: false,
            message: `You already created a new report today. You can create a new tomorrow.`,
            existingId: existing.id,
          };
        }
  }

  try {
    const crewName = toRequiredString(payload?.crewName);
    const crewNumber = toRequiredString(payload?.crewNumber);
    const officeResponsibleForFire = toRequiredString(payload?.officeResponsibleForFire);
    const fireName = toRequiredString(payload?.fireName);
    const fireNumber = toRequiredString(payload?.fireNumber);

    if (!crewName || !crewNumber || !officeResponsibleForFire || !fireName || !fireNumber) {
      return { success: false, message: "Missing required fields", data: null };
    }

    const timeEntries = normalizeTimeEntries(payload?.timeEntries);

    const reportInsert = {
      crewName,
      crewNumber,
      officeResponsibleForFire,
      fireName,
      fireNumber,
      form297Id: payload.form297Id,

      day1Date: payload?.day1Date || null,
      day2Date: payload?.day2Date || null,
      day3Date: payload?.day3Date || null,
      day4Date: payload?.day4Date || null,
      day5Date: payload?.day5Date || null,

      remarks: toOptionalString(payload?.remarks),

      officerInChargeSignature: toOptionalString(payload?.officerInChargeSignature),
      officerInChargeTitle: toOptionalString(payload?.officerInChargeTitle),

      postingPersonName: toOptionalString(payload?.postingPersonName),
      postingPersonSignature: toOptionalString(payload?.postingPersonSignature),
      postingDate: payload?.postingDate || null,

      additionalSignerName: toOptionalString(payload?.additionalSignerName),
      additionalSignerTitle: toOptionalString(payload?.additionalSignerTitle),
      additionalSignerSignature: toOptionalString(payload?.additionalSignerSignature),

      createdBy: user?.id || null,
      updatedBy: user?.id || null,
    };

    const result = await db.transaction(async (tx) => {
      let [form] = await tx
        .select()
        .from(allForms)
        .where(eq(allForms.formName, "form261"))
        .limit(1);

      if (!form) {
        const inserted = await tx.insert(allForms).values({
          formName: "form261",
          formDescription: "Form 261",
        });

        const insertId = inserted?.[0]?.insertId;
        form = { id: insertId };
      }


      console.log('reposrt: ', reportInsert.form297Id)
      const ids = await tx
        .insert(form261Reports)
        .values({ ...reportInsert, formId: form?.id || null, form297Id: reportInsert.form297Id })
        .$returningId();

      const reportId = ids?.[0]?.id;
      if (!reportId) return { reportId: null };

      if (timeEntries.length) {
        await tx
          .insert(form261TimeEntries)
          .values(timeEntries.map((e) => ({ ...e, reportId })));
      }

      return { reportId };
    });

    if (!result.reportId) {
      return { success: false, message: "Failed to create report", data: null };
    }

    return {
      success: true,
      message: "Report created successfully",
      data: { id: result.reportId },
    };
  } catch (e) {
    return {
      success: false,
      message: e?.message || "Failed to create report",
      data: null,
    };
  }
};

const updateForm261ReportService = async (payload, user) => {
  const id = payload?.id;
  if (!id || !Number.isInteger(id)) {
    return { success: false, message: "Invalid or missing report ID", data: null };
  }

  // if (!canAdminCreate(user)) {
  //   return { success: false, message: "Forbidden", data: null };
  // }

  try {
    const existing = await db
      .select()
      .from(form261Reports)
      .where(eq(form261Reports.id, id))
      .limit(1);

    if (!existing || existing.length === 0) {
      return { success: false, message: "Report not found", data: null };
    }

    if (existing[0].status !== "draft") {
      return {
        success: false,
        message: "Only draft reports can be updated",
        data: null,
      };
    }

    if (existing[0].createdBy && user?.id && existing[0].createdBy !== user.id && !user?.isAdmin) {
      return { success: false, message: "Forbidden", data: null };
    }

    const crewName = toRequiredString(payload?.crewName);
    const crewNumber = toRequiredString(payload?.crewNumber);
    const officeResponsibleForFire = toRequiredString(payload?.officeResponsibleForFire);
    const fireName = toRequiredString(payload?.fireName);
    const fireNumber = toRequiredString(payload?.fireNumber);

    if (!crewName || !crewNumber || !officeResponsibleForFire || !fireName || !fireNumber) {
      return { success: false, message: "Missing required fields", data: null };
    }

    const timeEntries = normalizeTimeEntries(payload?.timeEntries);

    const updateFields = {
      crewName,
      crewNumber,
      officeResponsibleForFire,
      fireName,
      fireNumber,
      form297Id: payload.form297Id,

      day1Date: payload?.day1Date || null,
      day2Date: payload?.day2Date || null,
      day3Date: payload?.day3Date || null,
      day4Date: payload?.day4Date || null,
      day5Date: payload?.day5Date || null,

      remarks: toOptionalString(payload?.remarks),

      officerInChargeSignature: toOptionalString(payload?.officerInChargeSignature),
      officerInChargeTitle: toOptionalString(payload?.officerInChargeTitle),

      postingPersonName: toOptionalString(payload?.postingPersonName),
      postingPersonSignature: toOptionalString(payload?.postingPersonSignature),
      postingDate: payload?.postingDate || null,

      additionalSignerName: toOptionalString(payload?.additionalSignerName),
      additionalSignerTitle: toOptionalString(payload?.additionalSignerTitle),
      additionalSignerSignature: toOptionalString(payload?.additionalSignerSignature),

      updatedBy: user?.id || null,
    };

    await db.transaction(async (tx) => {
      await tx.update(form261Reports).set(updateFields).where(eq(form261Reports.id, id));

      await tx
        .delete(form261TimeEntries)
        .where(eq(form261TimeEntries.reportId, id));

      if (timeEntries.length) {
        await tx
          .insert(form261TimeEntries)
          .values(timeEntries.map((e) => ({ ...e, reportId: id })));
      }
    });

    return { success: true, message: "Report updated successfully", data: null };
  } catch (e) {
    return {
      success: false,
      message: e?.message || "Failed to update report",
      data: null,
    };
  }
};

const submitForm261ReportService = async (id, user) => {
  if (!id || !Number.isInteger(id)) {
    return { success: false, message: "Invalid or missing report ID", data: null };
  }

  // if (!canAdminCreate(user)) {
  //   return { success: false, message: "Forbidden", data: null };
  // }

  try {
    const existing = await db
      .select()
      .from(form261Reports)
      .where(eq(form261Reports.id, id))
      .limit(1);

    if (!existing || existing.length === 0) {
      return { success: false, message: "Report not found", data: null };
    }

    if (existing[0].status !== "draft") {
      return { success: false, message: "Only draft reports can be submitted", data: null };
    }

    if (existing[0].createdBy && user?.id && existing[0].createdBy !== user.id && !user?.isAdmin) {
      return { success: false, message: "Forbidden", data: null };
    }

    await db
      .update(form261Reports)
      .set({
        status: "submitted",
        submittedAt: new Date(),
        updatedBy: user?.id || null,
      })
      .where(eq(form261Reports.id, id));

    return { success: true, message: "Report submitted", data: null };
  } catch (e) {
    return { success: false, message: e?.message || "Failed to submit report", data: null };
  }
};

const completeForm261ReportService = async (id, user) => {
  if (!user?.isAdmin) {
    return { success: false, message: "Forbidden! Only admin can do", data: null };
  }

  if (!id || !Number.isInteger(id)) {
    return { success: false, message: "Invalid or missing report ID", data: null };
  }

  try {
    const existing = await db
      .select()
      .from(form261Reports)
      .where(eq(form261Reports.id, id))
      .limit(1);

    if (!existing || existing.length === 0) {
      return { success: false, message: "Report not found", data: null };
    }

    if (existing[0].status !== "submitted") {
      return { success: false, message: "Only submitted reports can be completed", data: null };
    }

    await db
      .update(form261Reports)
      .set({
        status: "completed",
        completedAt: new Date(),
        updatedBy: user?.id || null,
      })
      .where(eq(form261Reports.id, id));

    return { success: true, message: "Report completed", data: null };
  } catch (e) {
    return { success: false, message: e?.message || "Failed to complete report", data: null };
  }
};

const deleteForm261ReportService = async (id, user) => {
  if (!id || !Number.isInteger(id)) {
    return { success: false, message: "Invalid or missing report ID", data: null };
  }

  try {
    const existing = await db
      .select()
      .from(form261Reports)
      .where(eq(form261Reports.id, id))
      .limit(1);

    if (!existing || existing.length === 0) {
      return { success: false, message: "Report not found", data: null };
    }

    const canDelete = existing[0].status === "draft" || user?.isAdmin;
    if (!canDelete) {
      return { success: false, message: "Only draft reports can be deleted", data: null };
    }

    if (existing[0].createdBy && user?.id && existing[0].createdBy !== user.id && !user?.isAdmin) {
      return { success: false, message: "Forbidden", data: null };
    }

    await db.delete(form261Reports).where(eq(form261Reports.id, id));

    return { success: true, message: "Report deleted", data: null };
  } catch (e) {
    return { success: false, message: e?.message || "Failed to delete report", data: null };
  }
};

const getForm261TicketsByCrewNumberAndDateRangeService = async (crewNumber, startDate, endDate, user) => {


  const isAdmin = user?.isAdmin;
  const userId = user?.id;

  if (!isAdmin && !userId) {
    return { success: false, message: "Unauthorized", data: null };
  }


  if (!crewNumber || !startDate || !endDate) {
    return {
      success: false,
      message: "CrewNumber, start date, and end date are required",
      data: null,
    };
  }


  try {

    const start = new Date(startDate);
    start.setHours(0, 0, 0, 0);

    const end = new Date(endDate);
    end.setHours(23, 59, 59, 999);

    const conditions = [
      eq(form261Reports.crewNumber, crewNumber),
      gte(form261Reports.createdAt, start),
      lte(form261Reports.createdAt, end)
    ]


    if (!isAdmin) {
      conditions.push(eq(form297Tickets.createdBy, userId));
    }


    const results = await db.select({
      report: form261Reports,
      timeEntries: form261TimeEntries

    }).from(form261Reports).innerJoin(form261TimeEntries, eq(form261TimeEntries.reportId, form261Reports.id)).where(and(...conditions));


    // console.log('result', results);

    return {
      success: true,
      message: "Reports fetched successfully",
      data: results,
    };

  }
  catch (e) {
    return {
      success: false,
      message: e.message,
      data: null,

    }
  }

}

module.exports = {
  getAllForm261ReportsService,
  getForm261ReportByIdService,
  createForm261ReportService,
  updateForm261ReportService,
  submitForm261ReportService,
  completeForm261ReportService,
  deleteForm261ReportService,
  getForm261TicketsByCrewNumberAndDateRangeService
};
