const { eq } = require("drizzle-orm");
const { db } = require("../db/index");
const { allForms } = require("../db/schemas/allForms");

// Get all forms
const getAllFormsService = async () => {
  try {
    const allFormsData = await db.select().from(allForms);

    if (allFormsData.length === 0) {
      return {
        success: false,
        message: "No forms found",
        data: null,
      };
    }

    return {
      success: true,
      message: "Forms fetched successfully",
      data: allFormsData,
    };
  } catch (e) {
    return {
      success: false,
      message: "Failed to fetch forms",
      data: null,
    };
  }
};

// Get form by ID
const getFormByIdService = async (formId) => {
  if (!formId || !Number.isInteger(formId)) {
    return {
      success: false,
      message: "Invalid or missing form ID",
      data: null,
    };
  }

  try {
    const formData = await db.select().from(allForms).where(eq(allForms.id, formId));

    if (formData.length === 0) {
      return {
        success: false,
        message: "No form found with the given ID",
        data: null,
      };
    }

    return {
      success: true,
      message: "Form fetched successfully",
      data: formData[0],
    };
  } catch (e) {
    return {
      success: false,
      message: "Failed to fetch form",
      data: null,
    };
  }
};

// Create new form
const createFormService = async (formData) => {
  try {
    const { formName, formDescription } = formData;

    if (!formName || formName.trim() === "") {
      return {
        success: false,
        message: "Form name is required",
        data: null,
      };
    }

    const result = await db.insert(allForms).values({
      formName: formName.trim(),
      formDescription: formDescription ? formDescription.trim() : null,
    }).$returningId();

    if (result.length === 0 || !result[0].id) {
      return {
        success: false,
        message: "Failed to create form",
        data: null,
      };
    }

    return {
      success: true,
      message: "Form created successfully",
      data: result[0],
    };
  } catch (e) {
    return {
      success: false,
      message: e.cause?.sqlMessage || e.message || "Failed to create form",
      data: null,
    };
  }
};

// Update form
const updateFormService = async (formId, formData) => {
  if (!formId || !Number.isInteger(formId)) {
    return {
      success: false,
      message: "Invalid or missing form ID",
      data: null,
    };
  }

  try {
    const { formName, formDescription } = formData;

    if (!formName || formName.trim() === "") {
      return {
        success: false,
        message: "Form name is required",
        data: null,
      };
    }

    const updateData = {
      formName: formName.trim(),
      formDescription: formDescription ? formDescription.trim() : null,
    };

    const result = await db
      .update(allForms)
      .set(updateData)
      .where(eq(allForms.id, formId));

    if (result.affectedRows === 0) {
      return {
        success: false,
        message: "No form found with the given ID",
        data: null,
      };
    }

    return {
      success: true,
      message: "Form updated successfully",
      data: null,
    };
  } catch (e) {
    return {
      success: false,
      message: e.cause?.sqlMessage || e.message || "Failed to update form",
      data: null,
    };
  }
};

// Delete form
const deleteFormService = async (formId) => {
  if (!formId || !Number.isInteger(formId)) {
    return {
      success: false,
      message: "Invalid or missing form ID",
      data: null,
    };
  }

  try {
    const result = await db.delete(allForms).where(eq(allForms.id, formId));

    if (result.affectedRows === 0) {
      return {
        success: false,
        message: "No form found with the given ID",
        data: null,
      };
    }

    return {
      success: true,
      message: "Form deleted successfully",
      data: null,
    };
  } catch (e) {
    return {
      success: false,
      message: e.cause?.sqlMessage || e.message || "Failed to delete form",
      data: null,
    };
  }
};

module.exports = {
  getAllFormsService,
  getFormByIdService,
  createFormService,
  updateFormService,
  deleteFormService,
};
