const express = require("express");

const {
  getAllFormsController,
  getFormByIdController,
  createFormController,
  updateFormController,
  deleteFormController,
} = require("../controllers/allFormsControllers");

const router = express.Router();

/**
 * @swagger
 * components:
 *   schemas:
 *     Form:
 *       type: object
 *       required:
 *         - formName
 *       properties:
 *         id:
 *           type: integer
 *           description: The auto-generated id of the form
 *         formName:
 *           type: string
 *           description: The name of the form
 *         formDescription:
 *           type: string
 *           description: The description of the form
 *       example:
 *         id: 1
 *         formName: "Incident Report Form"
 *         formDescription: "Form for reporting fire incidents"
 */

/**
 * @swagger
 * /api/all-forms:
 *   get:
 *     summary: Retrieve a list of all forms
 *     tags: [Forms]
 *     responses:
 *       200:
 *         description: A list of forms
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                   example: true
 *                 data:
 *                   type: array
 *                   items:
 *                     $ref: '#/components/schemas/Form'
 *                 message:
 *                   type: string
 *                   example: Forms fetched successfully
 *       500:
 *         description: Internal server error
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                   example: false
 *                 error:
 *                   type: string
 *                   example: Failed to fetch forms
 */
router.get("/", getAllFormsController);

/**
 * @swagger
 * /api/all-forms/{id}:
 *   get:
 *     summary: Get a form by ID
 *     tags: [Forms]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: The form ID
 *     responses:
 *       200:
 *         description: Form data
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                   example: true
 *                 data:
 *                   $ref: '#/components/schemas/Form'
 *                 message:
 *                   type: string
 *                   example: Form fetched successfully
 *       400:
 *         description: Invalid form ID
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                   example: false
 *                 error:
 *                   type: string
 *                   example: Invalid or missing form ID
 *       500:
 *         description: Internal server error
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                   example: false
 *                 error:
 *                   type: string
 *                   example: Failed to fetch form
 */
router.get("/:id", getFormByIdController);

/**
 * @swagger
 * /api/all-forms:
 *   post:
 *     summary: Create a new form
 *     tags: [Forms]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             required:
 *               - formName
 *             properties:
 *               formName:
 *                 type: string
 *                 description: The name of the form
 *               formDescription:
 *                 type: string
 *                 description: The description of the form
 *             example:
 *               formName: "Incident Report Form"
 *               formDescription: "Form for reporting fire incidents"
 *     responses:
 *       200:
 *         description: Form created successfully
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                   example: true
 *                 data:
 *                   type: object
 *                   properties:
 *                     id:
 *                       type: integer
 *                       example: 1
 *                 message:
 *                   type: string
 *                   example: Form created successfully
 *       400:
 *         description: Bad request - missing required fields or invalid data
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                   example: false
 *                 error:
 *                   type: string
 *                   example: Form name is required
 *       500:
 *         description: Internal server error
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                   example: false
 *                 error:
 *                   type: string
 *                   example: Failed to create form
 */
router.post("/", createFormController);

/**
 * @swagger
 * /api/all-forms/{id}:
 *   put:
 *     summary: Update an existing form
 *     tags: [Forms]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: The form ID
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             required:
 *               - formName
 *             properties:
 *               formName:
 *                 type: string
 *                 description: The name of the form
 *               formDescription:
 *                 type: string
 *                 description: The description of the form
 *             example:
 *               formName: "Updated Incident Report Form"
 *               formDescription: "Updated form for reporting fire incidents"
 *     responses:
 *       200:
 *         description: Form updated successfully
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                   example: true
 *                 data:
 *                   type: null
 *                 message:
 *                   type: string
 *                   example: Form updated successfully
 *       400:
 *         description: Bad request - invalid ID or missing required fields
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                   example: false
 *                 error:
 *                   type: string
 *                   example: Invalid or missing form ID
 *       500:
 *         description: Internal server error
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                   example: false
 *                 error:
 *                   type: string
 *                   example: Failed to update form
 */
router.put("/:id", updateFormController);

/**
 * @swagger
 * /api/all-forms/{id}:
 *   delete:
 *     summary: Delete a form
 *     tags: [Forms]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: The form ID
 *     responses:
 *       200:
 *         description: Form deleted successfully
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                   example: true
 *                 data:
 *                   type: null
 *                 message:
 *                   type: string
 *                   example: Form deleted successfully
 *       400:
 *         description: Invalid form ID
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                   example: false
 *                 error:
 *                   type: string
 *                   example: Invalid or missing form ID
 *       500:
 *         description: Internal server error
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                   example: false
 *                 error:
 *                   type: string
 *                   example: Failed to delete form
 */
router.delete("/:id", deleteFormController);

module.exports = router;