const { and, desc, eq, isNull, sql } = require("drizzle-orm");

const { db } = require("../db/index");
const { notifications } = require("../db/schemas/notificationsSchema");
const { httpStatus, responseTemplate } = require("../utils/responseHandler");

const parseId = (value) => {
  if (value === null || value === undefined) return null;
  const n = typeof value === "number" ? value : parseInt(String(value), 10);
  return Number.isInteger(n) ? n : null;
};

const parseBool = (value) => {
  const v = String(value || "").toLowerCase();
  return v === "true" || v === "1" || v === "yes";
};

const clampInt = (raw, { min, max, fallback }) => {
  const n = parseInt(String(raw ?? ""), 10);
  if (!Number.isFinite(n)) return fallback;
  return Math.max(min, Math.min(max, n));
};

const getNotificationsController = async (req, res) => {
  const userId = req?.user?.id;
  if (!userId) {
    return res
      .status(httpStatus.UNAUTHORIZED)
      .json(responseTemplate.error("Unauthorized"));
  }

  const unreadOnly = parseBool(req.query?.unread);
  const limit = clampInt(req.query?.limit, { min: 1, max: 200, fallback: 50 });
  const offset = clampInt(req.query?.offset, { min: 0, max: 1000000, fallback: 0 });

  try {
    const whereClause = unreadOnly
      ? and(eq(notifications.recipientId, userId), isNull(notifications.readAt))
      : eq(notifications.recipientId, userId);

    const rows = await db
      .select()
      .from(notifications)
      .where(whereClause)
      .orderBy(desc(notifications.id))
      .limit(limit)
      .offset(offset);

    return res
      .status(httpStatus.OK)
      .json(responseTemplate.success(rows, "Notifications fetched successfully"));
  } catch (e) {
    return res
      .status(httpStatus.INTERNAL_ERROR)
      .json(responseTemplate.error("Failed to fetch notifications", e.message));
  }
};

const getUnreadNotificationsCountController = async (req, res) => {
  const userId = req?.user?.id;
  if (!userId) {
    return res
      .status(httpStatus.UNAUTHORIZED)
      .json(responseTemplate.error("Unauthorized"));
  }

  try {
    const rows = await db
      .select({ count: sql`count(*)` })
      .from(notifications)
      .where(and(eq(notifications.recipientId, userId), isNull(notifications.readAt)));

    const rawCount = rows?.[0]?.count ?? 0;
    const count = typeof rawCount === "bigint" ? Number(rawCount) : Number(rawCount);

    return res
      .status(httpStatus.OK)
      .json(responseTemplate.success({ count }, "Unread count fetched successfully"));
  } catch (e) {
    return res
      .status(httpStatus.INTERNAL_ERROR)
      .json(responseTemplate.error("Failed to fetch unread count", e.message));
  }
};

const markNotificationReadController = async (req, res) => {
  const userId = req?.user?.id;
  const notificationId = parseId(req.params?.id);

  if (!userId) {
    return res
      .status(httpStatus.UNAUTHORIZED)
      .json(responseTemplate.error("Unauthorized"));
  }

  if (!notificationId) {
    return res
      .status(httpStatus.BAD_REQUEST)
      .json(responseTemplate.error("Invalid or missing notification ID"));
  }

  try {
    const result = await db
      .update(notifications)
      .set({ readAt: new Date() })
      .where(and(eq(notifications.id, notificationId), eq(notifications.recipientId, userId)));

    if (result.affectedRows === 0) {
      return res
        .status(httpStatus.NOT_FOUND)
        .json(responseTemplate.error("Notification not found"));
    }

    return res
      .status(httpStatus.OK)
      .json(responseTemplate.success(null, "Notification marked as read"));
  } catch (e) {
    return res
      .status(httpStatus.INTERNAL_ERROR)
      .json(responseTemplate.error("Failed to mark as read", e.message));
  }
};

const markNotificationUnreadController = async (req, res) => {
  const userId = req?.user?.id;
  const notificationId = parseId(req.params?.id);

  if (!userId) {
    return res
      .status(httpStatus.UNAUTHORIZED)
      .json(responseTemplate.error("Unauthorized"));
  }

  if (!notificationId) {
    return res
      .status(httpStatus.BAD_REQUEST)
      .json(responseTemplate.error("Invalid or missing notification ID"));
  }

  try {
    const result = await db
      .update(notifications)
      .set({ readAt: null })
      .where(and(eq(notifications.id, notificationId), eq(notifications.recipientId, userId)));

    if (result.affectedRows === 0) {
      return res
        .status(httpStatus.NOT_FOUND)
        .json(responseTemplate.error("Notification not found"));
    }

    return res
      .status(httpStatus.OK)
      .json(responseTemplate.success(null, "Notification marked as unread"));
  } catch (e) {
    return res
      .status(httpStatus.INTERNAL_ERROR)
      .json(responseTemplate.error("Failed to mark as unread", e.message));
  }
};

const markAllNotificationsReadController = async (req, res) => {
  const userId = req?.user?.id;

  if (!userId) {
    return res
      .status(httpStatus.UNAUTHORIZED)
      .json(responseTemplate.error("Unauthorized"));
  }

  try {
    const result = await db
      .update(notifications)
      .set({ readAt: new Date() })
      .where(and(eq(notifications.recipientId, userId), isNull(notifications.readAt)));

    return res
      .status(httpStatus.OK)
      .json(
        responseTemplate.success(
          { affectedRows: result.affectedRows },
          "All notifications marked as read",
        ),
      );
  } catch (e) {
    return res
      .status(httpStatus.INTERNAL_ERROR)
      .json(responseTemplate.error("Failed to mark all as read", e.message));
  }
};

const deleteNotificationController = async (req, res) => {
  const userId = req?.user?.id;
  const notificationId = parseId(req.params?.id);

  if (!userId) {
    return res
      .status(httpStatus.UNAUTHORIZED)
      .json(responseTemplate.error("Unauthorized"));
  }

  if (!notificationId) {
    return res
      .status(httpStatus.BAD_REQUEST)
      .json(responseTemplate.error("Invalid or missing notification ID"));
  }

  try {
    const result = await db
      .delete(notifications)
      .where(and(eq(notifications.id, notificationId), eq(notifications.recipientId, userId)));

    if (result.affectedRows === 0) {
      return res
        .status(httpStatus.NOT_FOUND)
        .json(responseTemplate.error("Notification not found"));
    }

    return res
      .status(httpStatus.OK)
      .json(responseTemplate.success(null, "Notification deleted successfully"));
  } catch (e) {
    return res
      .status(httpStatus.INTERNAL_ERROR)
      .json(responseTemplate.error("Failed to delete notification", e.message));
  }
};

module.exports = {
  getNotificationsController,
  getUnreadNotificationsCountController,
  markNotificationReadController,
  markNotificationUnreadController,
  markAllNotificationsReadController,
  deleteNotificationController,
};
