const {
  getAllForm297TicketsService,
  getForm297TicketByIdService,
  createForm297TicketService,
  updateForm297TicketService,
  submitForm297TicketService,
  deleteForm297TicketService,
  completeForm297TicketService,
  getForm297TicketsByAgreementAndDateRangeService,
} = require("../services/form297Services");

const { httpStatus, responseTemplate } = require("../utils/responseHandler");
const { db } = require("../db/index");
const { users } = require("../db/schemas/userSchema");
const { roles } = require("../db/schemas/roleSchema");
const { createNotificationsForRecipientsService } = require("../services/notificationsServices");
const { getIO } = require("../Socket/socket");
const { eq, inArray } = require("drizzle-orm");

const parseId = (value) => {
  if (value === null || value === undefined) return null;
  const n = typeof value === "number" ? value : parseInt(String(value), 10);
  return Number.isInteger(n) ? n : null;
};

const getAllForm297TicketsController = async (req, res) => {
  try {
    const result = await getAllForm297TicketsService(req.user);

    if (!result.success) {
      return res
        .status(httpStatus.OK)
        .json(responseTemplate.error("Could not fetch shift tickets", result.message));
    }

    return res
      .status(httpStatus.OK)
      .json(responseTemplate.success(result.data, result.message));
  } catch (e) {
    return res
      .status(httpStatus.INTERNAL_ERROR)
      .json(responseTemplate.error("Failed to fetch shift tickets", e.message));
  }
};

const getForm297TicketByIdController = async (req, res) => {
  const { id } = req.params;
  const ticketId = parseId(id);

  if (!ticketId) {
    return res
      .status(httpStatus.BAD_REQUEST)
      .json(responseTemplate.error("Invalid or missing ticket ID"));
  }

  try {
    const result = await getForm297TicketByIdService(ticketId, req.user);

    if (!result.success) {
      return res
        .status(httpStatus.BAD_REQUEST)
        .json(responseTemplate.error(result.message, result.data));
    }

    return res
      .status(httpStatus.OK)
      .json(responseTemplate.success(result.data, result.message));
  } catch (e) {
    return res
      .status(httpStatus.INTERNAL_ERROR)
      .json(responseTemplate.error("Failed to fetch ticket", e.message));
  }
};

const createForm297TicketController = async (req, res) => {
  const {
    agreementNumber,
    contractorAgencyName,
    resourceOrderNumber,
    incidentName,
    incidentNumber,
    financialCode,
  } = req.body || {};

  if (
    !agreementNumber ||
    !contractorAgencyName ||
    !resourceOrderNumber ||
    !incidentName ||
    !incidentNumber ||
    !financialCode
  ) {
    return res
      .status(httpStatus.OK)
      .json(responseTemplate.error("Missing required fields"));
  }

  try {
    const userId = req?.user?.id;
    const result = await createForm297TicketService(req.body, userId);

    if (!result.success) {
      return res
        .status(httpStatus.OK)
        .json(responseTemplate.error(result.message, result.data));
    }

    return res
      .status(httpStatus.OK)
      .json(responseTemplate.success(result.data, result.message));
  } catch (e) {
    return res
      .status(httpStatus.INTERNAL_ERROR)
      .json(responseTemplate.error("Failed to create ticket", e.message));
  }
};

const updateForm297TicketController = async (req, res) => {
  const {
    id: rawId,
    agreementNumber,
    contractorAgencyName,
    resourceOrderNumber,
    incidentName,
    incidentNumber,
    financialCode,
  } = req.body || {};

  const id = parseId(rawId);

  if (!id) {
    return res
      .status(httpStatus.BAD_REQUEST)
      .json(responseTemplate.error("Invalid or missing ticket ID"));
  }

  if (
    !agreementNumber ||
    !contractorAgencyName ||
    !resourceOrderNumber ||
    !incidentName ||
    !incidentNumber ||
    !financialCode
  ) {
    return res
      .status(httpStatus.BAD_REQUEST)
      .json(responseTemplate.error("Missing required fields"));
  }

  try {
    const userId = req?.user?.id;
    const result = await updateForm297TicketService({ ...req.body, id }, req.user);

    if (!result.success) {
      return res
        .status(httpStatus.OK)
        .json(responseTemplate.error(result.message, result.data));
    }

    return res
      .status(httpStatus.OK)
      .json(responseTemplate.success(result.data, result.message));
  } catch (e) {
    return res
      .status(httpStatus.INTERNAL_ERROR)
      .json(responseTemplate.error("Failed to update ticket", e.message));
  }
};

const submitForm297TicketController = async (req, res) => {
  const { id } = req.params;
  const ticketId = parseId(id);

  if (!ticketId) {
    return res
      .status(httpStatus.OK)
      .json(responseTemplate.error("Invalid or missing ticket ID"));
  }

  try {
    const userId = req?.user?.id;
    const result = await submitForm297TicketService(ticketId, req.user);

    if (!result.success) {
      return res
        .status(httpStatus.OK)
        .json(responseTemplate.error(result.message, result.data));
    }

    // Fire-and-forget notifications (do not block submit if notifications fail)
    try {
      const superAdminRoles = await db
        .select({ id: roles.id })
        .from(roles)
        .where(eq(roles.isSuperAdmin, true));

      const roleIds = Array.from(
        new Set(
          [...(superAdminRoles || []).map((r) => r.id)].filter((v) => Number.isInteger(v)),
        ),
      );

      if (roleIds.length > 0) {
        const recipients = await db
          .select({ id: users.id })
          .from(users)
          .where(inArray(users.role, roleIds));

        const recipientIds = (recipients || [])
          .map((u) => u.id)
          .filter((v) => Number.isInteger(v));

        if (recipientIds.length > 0) {
          await createNotificationsForRecipientsService(
            {
              entityType: "form297",
              entityId: String(ticketId),
              type: "form297_submitted",
              title: "Form 297 Submitted",
              message: `Ticket #${ticketId} has been submitted`,
              payload: { ticketId },
            },
            recipientIds,
            { senderId: userId },
          );

          const io = getIO();
          recipientIds.forEach((rid) => {
            io.to(`user:${rid}`).emit("notification:new", {
              entityType: "form297",
              entityId: String(ticketId),
              type: "form297_submitted",
              ticketId,
            });
          });
        }
      }
    } catch (e) {
      // ignore
    }

    return res
      .status(httpStatus.OK)
      .json(responseTemplate.success(result.data, result.message));
  } catch (e) {
    return res
      .status(httpStatus.INTERNAL_ERROR)
      .json(responseTemplate.error("Failed to submit ticket", e.message));
  }
};


const completeForm297TicketController = async (req, res) => {
  const { id } = req.params;
  const ticketId = parseId(id);

  if (!ticketId) {
    return res
      .status(httpStatus.OK)
      .json(responseTemplate.error("Invalid or missing ticket ID"));
  }

  try {
    // const userId = req?.user?.id;
    const result = await completeForm297TicketService(ticketId, req.user);

    if (!result.success) {
      return res
        .status(httpStatus.OK)
        .json(responseTemplate.error(result.message, result.data));
    }

    return res
      .status(httpStatus.OK)
      .json(responseTemplate.success(result.data, result.message));
  } catch (e) {
    console.log(e);
    return res
      .status(httpStatus.INTERNAL_ERROR)
      .json(responseTemplate.error("Failed to complete ticket", e.message));
  }
};

// const supervisorSignForm297TicketController = async (req, res) => {
//   const { id } = req.params;
//   const ticketId = parseId(id);

//   if (!ticketId) {
//     return res
//       .status(httpStatus.BAD_REQUEST)
//       .json(responseTemplate.error("Invalid or missing ticket ID"));
//   }

//   try {
//     const userId = req?.user?.id;
//     const result = await supervisorSignForm297TicketService(ticketId, req.body, userId);

//     if (!result.success) {
//       return res
//         .status(httpStatus.BAD_REQUEST)
//         .json(responseTemplate.error(result.message, result.data));
//     }

//     return res
//       .status(httpStatus.OK)
//       .json(responseTemplate.success(result.data, result.message));
//   } catch (e) {
//     return res
//       .status(httpStatus.INTERNAL_ERROR)
//       .json(responseTemplate.error("Failed to sign ticket", e.message));
//   }
// };

const deleteForm297TicketController = async (req, res) => {
  const { id } = req.params;
  const ticketId = parseId(id);

  if (!ticketId) {
    return res
      .status(httpStatus.BAD_REQUEST)
      .json(responseTemplate.error("Invalid or missing ticket ID"));
  }

  try {
    const userId = req?.user?.id;
    const result = await deleteForm297TicketService(ticketId, userId , req.user);

    if (!result.success) {
      return res
        .status(httpStatus.OK)
        .json(responseTemplate.error(result.message, result.data));
    }

    return res
      .status(httpStatus.OK)
      .json(responseTemplate.success(result.data, result.message));
  } catch (e) {
    return res
      .status(httpStatus.INTERNAL_ERROR)
      .json(responseTemplate.error("Failed to delete ticket", e.message));
  }
};

const getForm297TicketsByAgreementAndDateRangeController = async (req, res) => {
  const { agreementNumber, startDate, endDate } = req.query;

  if (!agreementNumber || !startDate || !endDate) {
    return res
      .status(httpStatus.OK)
      .json(responseTemplate.error("Agreement number, start date, and end date are required"));
  }

  try {
    const result = await getForm297TicketsByAgreementAndDateRangeService(
      agreementNumber,
      startDate,
      endDate,
      req.user
    );

    if (!result.success) {
      return res
        .status(httpStatus.OK)
        .json(responseTemplate.error(result.message, result.data));
    }

    return res
      .status(httpStatus.OK)
      .json(responseTemplate.success(result.data, result.message));
  } catch (e) {
    return res
      .status(httpStatus.INTERNAL_ERROR)
      .json(responseTemplate.error("Failed to fetch tickets", e.message));
  }
};

module.exports = {
  getAllForm297TicketsController,
  getForm297TicketByIdController,
  createForm297TicketController,
  updateForm297TicketController,
  submitForm297TicketController,
  completeForm297TicketController,
  deleteForm297TicketController,
  getForm297TicketsByAgreementAndDateRangeController,
};
