const {
  getAllForm288ReportsService,
  getForm288ReportByIdService,
  createForm288ReportService,
  updateForm288ReportService,
  submitForm288ReportService,
  completeForm288ReportService,
  deleteForm288ReportService,
} = require("../services/form288Services");

const { httpStatus, responseTemplate } = require("../utils/responseHandler");

const parseId = (value) => {
  if (value === null || value === undefined) return null;
  const n = typeof value === "number" ? value : parseInt(String(value), 10);
  return Number.isInteger(n) ? n : null;
};

const getAllForm288ReportsController = async (req, res) => {
  try {
    const result = await getAllForm288ReportsService(req.user);

    if (!result.success) {
      return res
        .status(httpStatus.OK)
        .json(responseTemplate.error("Could not fetch reports", result.message));
    }

    return res
      .status(httpStatus.OK)
      .json(responseTemplate.success(result.data, result.message));
  } catch (e) {
    return res
      .status(httpStatus.INTERNAL_ERROR)
      .json(responseTemplate.error("Failed to fetch reports", e.message));
  }
};

const getForm288ReportByIdController = async (req, res) => {
  const { id } = req.params;
  const reportId = parseId(id);

  if (!reportId) {
    return res
      .status(httpStatus.BAD_REQUEST)
      .json(responseTemplate.error("Invalid or missing report ID"));
  }

  try {
    const result = await getForm288ReportByIdService(reportId, req.user);

    if (!result.success) {
      return res
        .status(httpStatus.BAD_REQUEST)
        .json(responseTemplate.error(result.message, result.data));
    }

    return res
      .status(httpStatus.OK)
      .json(responseTemplate.success(result.data, result.message));
  } catch (e) {
    return res
      .status(httpStatus.INTERNAL_ERROR)
      .json(responseTemplate.error("Failed to fetch report", e.message));
  }
};

const createForm288ReportController = async (req, res) => {
  try {
    const result = await createForm288ReportService(req.body, req.user);

    if (!result.success) {
      const status = result.message === "Forbidden" ? httpStatus.FORBIDDEN : httpStatus.BAD_REQUEST;
      return res.status(status).json(responseTemplate.error(result.message, result.data));
    }

    return res
      .status(httpStatus.OK)
      .json(responseTemplate.success(result.data, result.message));
  } catch (e) {
    return res
      .status(httpStatus.INTERNAL_ERROR)
      .json(responseTemplate.error("Failed to create report", e.message));
  }
};

const updateForm288ReportController = async (req, res) => {
  const { id: rawId } = req.body || {};
  const id = parseId(rawId);

  if (!id) {
    return res
      .status(httpStatus.BAD_REQUEST)
      .json(responseTemplate.error("Invalid or missing report ID"));
  }

  try {
    const result = await updateForm288ReportService({ ...req.body, id }, req.user);

    if (!result.success) {
      const status = result.message === "Forbidden" ? httpStatus.FORBIDDEN : httpStatus.BAD_REQUEST;
      return res.status(status).json(responseTemplate.error(result.message, result.data));
    }

    return res
      .status(httpStatus.OK)
      .json(responseTemplate.success(result.data, result.message));
  } catch (e) {
    return res
      .status(httpStatus.INTERNAL_ERROR)
      .json(responseTemplate.error("Failed to update report", e.message));
  }
};

const submitForm288ReportController = async (req, res) => {
  const { id } = req.params;
  const reportId = parseId(id);

  if (!reportId) {
    return res
      .status(httpStatus.BAD_REQUEST)
      .json(responseTemplate.error("Invalid or missing report ID"));
  }

  try {
    const result = await submitForm288ReportService(reportId, req.user);

    if (!result.success) {
      const status = result.message === "Forbidden" ? httpStatus.FORBIDDEN : httpStatus.BAD_REQUEST;
      return res.status(status).json(responseTemplate.error(result.message, result.data));
    }

    return res
      .status(httpStatus.OK)
      .json(responseTemplate.success(result.data, result.message));
  } catch (e) {
    return res
      .status(httpStatus.INTERNAL_ERROR)
      .json(responseTemplate.error("Failed to submit report", e.message));
  }
};

const completeForm288ReportController = async (req, res) => {
  const { id } = req.params;
  const reportId = parseId(id);

  if (!reportId) {
    return res
      .status(httpStatus.BAD_REQUEST)
      .json(responseTemplate.error("Invalid or missing report ID"));
  }

  try {
    const result = await completeForm288ReportService(reportId, req.user);

    if (!result.success) {
      const status = result.message === "Forbidden" ? httpStatus.FORBIDDEN : httpStatus.BAD_REQUEST;
      return res.status(status).json(responseTemplate.error(result.message, result.data));
    }

    return res
      .status(httpStatus.OK)
      .json(responseTemplate.success(result.data, result.message));
  } catch (e) {
    return res
      .status(httpStatus.INTERNAL_ERROR)
      .json(responseTemplate.error("Failed to complete report", e.message));
  }
};

const deleteForm288ReportController = async (req, res) => {
  const { id } = req.params;
  const reportId = parseId(id);

  if (!reportId) {
    return res
      .status(httpStatus.BAD_REQUEST)
      .json(responseTemplate.error("Invalid or missing report ID"));
  }

  try {
    const result = await deleteForm288ReportService(reportId, req.user);

    if (!result.success) {
      const status = result.message === "Forbidden" ? httpStatus.FORBIDDEN : httpStatus.BAD_REQUEST;
      return res.status(status).json(responseTemplate.error(result.message, result.data));
    }

    return res
      .status(httpStatus.OK)
      .json(responseTemplate.success(result.data, result.message));
  } catch (e) {
    return res
      .status(httpStatus.INTERNAL_ERROR)
      .json(responseTemplate.error("Failed to delete report", e.message));
  }
};

module.exports = {
  getAllForm288ReportsController,
  getForm288ReportByIdController,
  createForm288ReportController,
  updateForm288ReportController,
  submitForm288ReportController,
  completeForm288ReportController,
  deleteForm288ReportController,
};
