const {
  getAllForm268ReportsService,
  getForm268ReportByIdService,
  createForm268ReportService,
  updateForm268ReportService,
  submitForm268ReportService,
  completeForm268ReportService,
  deleteForm268ReportService,
} = require("../services/form268Services");

const { httpStatus, responseTemplate } = require("../utils/responseHandler");

const parseId = (value) => {
  if (value === null || value === undefined) return null;
  const n = typeof value === "number" ? value : parseInt(String(value), 10);
  return Number.isInteger(n) ? n : null;
};

const getAllForm268ReportsController = async (req, res) => {
  try {
    const result = await getAllForm268ReportsService(req.user);

    if (!result.success) {
      return res
        .status(httpStatus.OK)
        .json(responseTemplate.error("Could not fetch reports", result.message));
    }

    return res
      .status(httpStatus.OK)
      .json(responseTemplate.success(result.data, result.message));
  } catch (e) {
    return res
      .status(httpStatus.INTERNAL_ERROR)
      .json(responseTemplate.error("Failed to fetch reports", e.message));
  }
};

const getForm268ReportByIdController = async (req, res) => {
  const { id } = req.params;
  const reportId = parseId(id);

  if (!reportId) {
    return res
      .status(httpStatus.BAD_REQUEST)
      .json(responseTemplate.error("Invalid or missing report ID"));
  }

  try {
    const result = await getForm268ReportByIdService(reportId, req.user);

    if (!result.success) {
      return res
        .status(httpStatus.BAD_REQUEST)
        .json(responseTemplate.error(result.message, result.data));
    }

    return res
      .status(httpStatus.OK)
      .json(responseTemplate.success(result.data, result.message));
  } catch (e) {
    return res
      .status(httpStatus.INTERNAL_ERROR)
      .json(responseTemplate.error("Failed to fetch report", e.message));
  }
};

const createForm268ReportController = async (req, res) => {
  try {
    const result = await createForm268ReportService(req.body, req.user);

    if (!result.success) {
      const status = result.message === "Forbidden" ? httpStatus.FORBIDDEN : httpStatus.BAD_REQUEST;
      return res.status(status).json(responseTemplate.error(result.message, result.data));
    }

    return res
      .status(httpStatus.OK)
      .json(responseTemplate.success(result.data, result.message));
  } catch (e) {
    return res
      .status(httpStatus.INTERNAL_ERROR)
      .json(responseTemplate.error("Failed to create report", e.message));
  }
};

const updateForm268ReportController = async (req, res) => {
  const { id: rawId } = req.body || {};
  const id = parseId(rawId);

  if (!id) {
    return res
      .status(httpStatus.BAD_REQUEST)
      .json(responseTemplate.error("Invalid or missing report ID"));
  }

  try {
    const result = await updateForm268ReportService({ ...req.body, id }, req.user);

    if (!result.success) {
      const status = result.message === "Forbidden" ? httpStatus.FORBIDDEN : httpStatus.BAD_REQUEST;
      return res.status(status).json(responseTemplate.error(result.message, result.data));
    }

    return res
      .status(httpStatus.OK)
      .json(responseTemplate.success(result.data, result.message));
  } catch (e) {
    return res
      .status(httpStatus.INTERNAL_ERROR)
      .json(responseTemplate.error("Failed to update report", e.message));
  }
};

const submitForm268ReportController = async (req, res) => {
  const { id } = req.params;
  const reportId = parseId(id);

  if (!reportId) {
    return res
      .status(httpStatus.BAD_REQUEST)
      .json(responseTemplate.error("Invalid or missing report ID"));
  }

  try {
    const result = await submitForm268ReportService(reportId, req.user);

    if (!result.success) {
      const status = result.message === "Forbidden" ? httpStatus.FORBIDDEN : httpStatus.BAD_REQUEST;
      return res.status(status).json(responseTemplate.error(result.message, result.data));
    }

    return res
      .status(httpStatus.OK)
      .json(responseTemplate.success(result.data, result.message));
  } catch (e) {
    return res
      .status(httpStatus.INTERNAL_ERROR)
      .json(responseTemplate.error("Failed to submit report", e.message));
  }
};

const completeForm268ReportController = async (req, res) => {
  const { id } = req.params;
  const reportId = parseId(id);

  if (!reportId) {
    return res
      .status(httpStatus.BAD_REQUEST)
      .json(responseTemplate.error("Invalid or missing report ID"));
  }

  try {
    const result = await completeForm268ReportService(reportId, req.user);

    if (!result.success) {
      const status = result.message === "Forbidden" ? httpStatus.FORBIDDEN : httpStatus.BAD_REQUEST;
      return res.status(status).json(responseTemplate.error(result.message, result.data));
    }

    return res
      .status(httpStatus.OK)
      .json(responseTemplate.success(result.data, result.message));
  } catch (e) {
    return res
      .status(httpStatus.INTERNAL_ERROR)
      .json(responseTemplate.error("Failed to complete report", e.message));
  }
};

const deleteForm268ReportController = async (req, res) => {
  const { id } = req.params;
  const reportId = parseId(id);

  if (!reportId) {
    return res
      .status(httpStatus.BAD_REQUEST)
      .json(responseTemplate.error("Invalid or missing report ID"));
  }

  try {
    const result = await deleteForm268ReportService(reportId, req.user);

    if (!result.success) {
      const status = result.message === "Forbidden" ? httpStatus.FORBIDDEN : httpStatus.BAD_REQUEST;
      return res.status(status).json(responseTemplate.error(result.message, result.data));
    }

    return res
      .status(httpStatus.OK)
      .json(responseTemplate.success(result.data, result.message));
  } catch (e) {
    return res
      .status(httpStatus.INTERNAL_ERROR)
      .json(responseTemplate.error("Failed to delete report", e.message));
  }
};

module.exports = {
  getAllForm268ReportsController,
  getForm268ReportByIdController,
  createForm268ReportController,
  updateForm268ReportController,
  submitForm268ReportController,
  completeForm268ReportController,
  deleteForm268ReportController,
};
