const {
  getAllForm261ReportsService,
  getForm261ReportByIdService,
  createForm261ReportService,
  updateForm261ReportService,
  submitForm261ReportService,
  completeForm261ReportService,
  deleteForm261ReportService,
  getForm261TicketsByCrewNumberAndDateRangeService,
} = require("../services/form261Services");

const { httpStatus, responseTemplate } = require("../utils/responseHandler");

const parseId = (value) => {
  if (value === null || value === undefined) return null;
  const n = typeof value === "number" ? value : parseInt(String(value), 10);
  return Number.isInteger(n) ? n : null;
};

const getAllForm261ReportsController = async (req, res) => {
  try {
    const result = await getAllForm261ReportsService(req.user);

    if (!result.success) {
      return res
        .status(httpStatus.OK)
        .json(responseTemplate.error("Could not fetch reports", result.message));
    }

    return res
      .status(httpStatus.OK)
      .json(responseTemplate.success(result.data, result.message));
  } catch (e) {
    return res
      .status(httpStatus.INTERNAL_ERROR)
      .json(responseTemplate.error("Failed to fetch reports", e.message));
  }
};

const getForm261ReportByIdController = async (req, res) => {
  const { id } = req.params;
  const reportId = parseId(id);

  if (!reportId) {
    return res
      .status(httpStatus.OK)
      .json(responseTemplate.error("Invalid or missing report ID"));
  }

  try {
    const result = await getForm261ReportByIdService(reportId, req.user);

    if (!result.success) {
      return res
        .status(httpStatus.OK)
        .json(responseTemplate.error(result.message, result.data));
    }

    return res
      .status(httpStatus.OK)
      .json(responseTemplate.success(result.data, result.message));
  } catch (e) {
    return res
      .status(httpStatus.INTERNAL_ERROR)
      .json(responseTemplate.error("Failed to fetch report", e.message));
  }
};

const createForm261ReportController = async (req, res) => {
  try {
    const result = await createForm261ReportService(req.body, req.user);

    if (!result.success) {
      const status = result.message === "Forbidden" ? httpStatus.FORBIDDEN : httpStatus.BAD_REQUEST;
      return res.status(status).json(responseTemplate.error(result.message, result.data));
    }

    return res
      .status(httpStatus.OK)
      .json(responseTemplate.success(result.data, result.message));
  } catch (e) {
    return res
      .status(httpStatus.OK)
      .json(responseTemplate.error("Failed to create report", e.message));
  }
};

const updateForm261ReportController = async (req, res) => {
  const { id: rawId } = req.body || {};
  const id = parseId(rawId);

  if (!id) {
    return res
      .status(httpStatus.OK)
      .json(responseTemplate.error("Invalid or missing report ID"));
  }

  try {
    const result = await updateForm261ReportService({ ...req.body, id }, req.user);

    if (!result.success) {
      const status = result.message === "Forbidden" ? httpStatus.FORBIDDEN : httpStatus.OK;
      return res.status(status).json(responseTemplate.error(result.message, result.data));
    }

    return res
      .status(httpStatus.OK)
      .json(responseTemplate.success(result.data, result.message));
  } catch (e) {
    return res
      .status(httpStatus.INTERNAL_ERROR)
      .json(responseTemplate.error("Failed to update report", e.message));
  }
};

const submitForm261ReportController = async (req, res) => {
  const { id } = req.params;
  const reportId = parseId(id);

  if (!reportId) {
    return res
      .status(httpStatus.OK)
      .json(responseTemplate.error("Invalid or missing report ID"));
  }

  try {
    const result = await submitForm261ReportService(reportId, req.user);

    if (!result.success) {
      const status = result.message === "Forbidden" ? httpStatus.FORBIDDEN : httpStatus.OK;
      return res.status(status).json(responseTemplate.error(result.message, result.data));
    }

    return res
      .status(httpStatus.OK)
      .json(responseTemplate.success(result.data, result.message));
  } catch (e) {
    return res
      .status(httpStatus.INTERNAL_ERROR)
      .json(responseTemplate.error("Failed to submit report", e.message));
  }
};

const completeForm261ReportController = async (req, res) => {
  const { id } = req.params;
  const reportId = parseId(id);

  if (!reportId) {
    return res
      .status(httpStatus.BAD_REQUEST)
      .json(responseTemplate.error("Invalid or missing report ID"));
  }

  try {
    const result = await completeForm261ReportService(reportId, req.user);

    if (!result.success) {
      const status = result.message === "Forbidden" ? httpStatus.FORBIDDEN : httpStatus.OK;
      return res.status(status).json(responseTemplate.error(result.message, result.data));
    }

    return res
      .status(httpStatus.OK)
      .json(responseTemplate.success(result.data, result.message));
  } catch (e) {
    return res
      .status(httpStatus.INTERNAL_ERROR)
      .json(responseTemplate.error("Failed to complete report", e.message));
  }
};

const deleteForm261ReportController = async (req, res) => {
  const { id } = req.params;
  const reportId = parseId(id);

  if (!reportId) {
    return res
      .status(httpStatus.BAD_REQUEST)
      .json(responseTemplate.error("Invalid or missing report ID"));
  }

  try {
    const result = await deleteForm261ReportService(reportId, req.user);

    if (!result.success) {
      const status = result.message === "Forbidden" ? httpStatus.FORBIDDEN : httpStatus.OK;
      return res.status(status).json(responseTemplate.error(result.message, result.data));
    }

    return res
      .status(httpStatus.OK)
      .json(responseTemplate.success(result.data, result.message));
  } catch (e) {
    return res
      .status(httpStatus.INTERNAL_ERROR)
      .json(responseTemplate.error("Failed to delete report", e.message));
  }
};



const getForm261TicketsByCrewNumberAndDateRangeController = async (req, res) => {

  const { crewNumber, startDate, endDate } = req.body;

  if (!crewNumber) {
    return res.status(httpStatus.BAD_REQUEST).json(responseTemplate.error("Crew Number is required!"))
  }
  else if (!startDate) {
    return  res.status(httpStatus.BAD_REQUEST).json(responseTemplate.error("StartDate is required!"))

  }
  else if(!endDate) {
    return res.status(httpStatus.BAD_REQUEST).json(responseTemplate.error("EndDate is required!"))

  }



  try {
    const result = await getForm261TicketsByCrewNumberAndDateRangeService(crewNumber, startDate, endDate , req.user);

    if (!result.success) {
      return res.status(httpStatus.OK).json(responseTemplate.error(result.message, result.data))
    }

    return res.status(httpStatus.OK).json(responseTemplate.success(result.data, result.message))



  } catch (e) {
    return res
      .status(httpStatus.INTERNAL_ERROR)
      .json(responseTemplate.error("Failed to fetch tickets", e.message));
  }

}
module.exports = {
  getAllForm261ReportsController,
  getForm261ReportByIdController,
  createForm261ReportController,
  updateForm261ReportController,
  submitForm261ReportController,
  completeForm261ReportController,
  deleteForm261ReportController,
  getForm261TicketsByCrewNumberAndDateRangeController
};
