const {
  getAllFormsService,
  getFormByIdService,
  createFormService,
  updateFormService,
  deleteFormService,
} = require("../services/allFormsServices");

const { httpStatus, responseTemplate } = require("../utils/responseHandler");

const parseId = (value) => {
  if (value === null || value === undefined) return null;
  const n = typeof value === "number" ? value : parseInt(String(value), 10);
  return Number.isInteger(n) ? n : null;
};

const getAllFormsController = async (req, res) => {
  try {
    const result = await getAllFormsService();

    if (!result.success) {
      return res
        .status(httpStatus.OK)
        .json(responseTemplate.error("Could not fetch forms", result.message));
    }

    return res
      .status(httpStatus.OK)
      .json(responseTemplate.success(result.data, result.message));
  } catch (e) {
    return res
      .status(httpStatus.INTERNAL_ERROR)
      .json(responseTemplate.error("Failed to fetch forms", e.message));
  }
};

const getFormByIdController = async (req, res) => {
  const { id } = req.params;
  const formId = parseId(id);

  if (!formId) {
    return res
      .status(httpStatus.BAD_REQUEST)
      .json(responseTemplate.error("Invalid or missing form ID"));
  }

  try {
    const result = await getFormByIdService(formId);

    if (!result.success) {
      return res
        .status(httpStatus.BAD_REQUEST)
        .json(responseTemplate.error(result.message, result.data));
    }

    return res
      .status(httpStatus.OK)
      .json(responseTemplate.success(result.data, result.message));
  } catch (e) {
    return res
      .status(httpStatus.INTERNAL_ERROR)
      .json(responseTemplate.error("Failed to fetch form", e.message));
  }
};

const createFormController = async (req, res) => {
  const { formName, formDescription } = req.body;

  if (!formName || formName.trim() === "") {
    return res
      .status(httpStatus.BAD_REQUEST)
      .json(responseTemplate.error("Form name is required"));
  }

  try {
    const result = await createFormService(req.body);

    if (!result.success) {
      return res
        .status(httpStatus.BAD_REQUEST)
        .json(responseTemplate.error(result.message, result.data));
    }

    return res
      .status(httpStatus.OK)
      .json(responseTemplate.success(result.data, result.message));
  } catch (e) {
    return res
      .status(httpStatus.INTERNAL_ERROR)
      .json(responseTemplate.error("Failed to create form", e.message));
  }
};

const updateFormController = async (req, res) => {
  const { id } = req.params;
  const formId = parseId(id);

  if (!formId) {
    return res
      .status(httpStatus.BAD_REQUEST)
      .json(responseTemplate.error("Invalid or missing form ID"));
  }

  const { formName, formDescription } = req.body;

  if (!formName || formName.trim() === "") {
    return res
      .status(httpStatus.BAD_REQUEST)
      .json(responseTemplate.error("Form name is required"));
  }

  try {
    const result = await updateFormService(formId, req.body);

    if (!result.success) {
      return res
        .status(httpStatus.BAD_REQUEST)
        .json(responseTemplate.error(result.message, result.data));
    }

    return res
      .status(httpStatus.OK)
      .json(responseTemplate.success(result.data, result.message));
  } catch (e) {
    return res
      .status(httpStatus.INTERNAL_ERROR)
      .json(responseTemplate.error("Failed to update form", e.message));
  }
};

const deleteFormController = async (req, res) => {
  const { id } = req.params;
  const formId = parseId(id);

  if (!formId) {
    return res
      .status(httpStatus.BAD_REQUEST)
      .json(responseTemplate.error("Invalid or missing form ID"));
  }

  try {
    const result = await deleteFormService(formId);

    if (!result.success) {
      return res
        .status(httpStatus.BAD_REQUEST)
        .json(responseTemplate.error(result.message, result.data));
    }

    return res
      .status(httpStatus.OK)
      .json(responseTemplate.success(result.data, result.message));
  } catch (e) {
    return res
      .status(httpStatus.INTERNAL_ERROR)
      .json(responseTemplate.error("Failed to delete form", e.message));
  }
};

module.exports = {
  getAllFormsController,
  getFormByIdController,
  createFormController,
  updateFormController,
  deleteFormController,
};